<?php
class AdminController extends Controller {
    private $userModel;
    private $examModel;
    private $classModel;

    public function __construct() {
        AuthMiddleware::check(['admin']);
        $this->userModel = $this->model('User');
        $this->examModel = $this->model('Exam');
        $this->classModel = $this->model('ClassModel');
    }

    public function dashboard() {
        // Fetch Stats using models
        $stats = [
            'siswa' => $this->userModel->countByRole('siswa'),
            'guru' => $this->userModel->countByRole('guru'),
            'exams_active' => $this->examModel->countExamsByStatus('published'),
            'exams_completed' => $this->examModel->countCompletedSessions()
        ];
        
        // Fetch exam average scores for chart
        $chart_data = $this->examModel->getAverageScoresByExam(7);

        $data = [
            'title' => 'Admin Dashboard',
            'active_menu' => 'dashboard',
            'stats' => $stats,
            'chart_data' => $chart_data
        ];
        $this->view('layouts/header', $data);
        $this->view('admin/dashboard', $data);
        $this->view('layouts/footer');
    }

    public function users() {
        $users = $this->userModel->getAllUsers(); 
        $classes = $this->classModel->getAllClasses(); // Fetch classes for dropdown
        $data = [
            'title' => 'Manage Users',
            'active_menu' => 'users',
            'users' => $users,
            'classes' => $classes
        ];
        $this->view('layouts/header', $data);
        $this->view('admin/users', $data);
        $this->view('layouts/footer');
    }

    public function store_user() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $role = trim($_POST['role']);
            
            // Prepare data array
            $data = [
                'role' => $role,
                'password' => trim($_POST['password']),
                'nisn' => null,
                'tempat_lahir' => null,
                'tanggal_lahir' => null,
                'jenis_kelamin' => null,
                'alamat' => null,
                'telepon' => null,
                'foto' => null,
                'class_id' => null
            ];
            
            // Handle Student (Siswa) - Use NISN as username
            if ($role == 'siswa') {
                $data['nisn'] = trim($_POST['nisn']);
                $data['username'] = $data['nisn']; // NISN becomes username
                $data['name'] = trim($_POST['name']);
                $data['tempat_lahir'] = isset($_POST['tempat_lahir']) ? trim($_POST['tempat_lahir']) : null;
                $data['tanggal_lahir'] = !empty($_POST['tanggal_lahir']) ? $_POST['tanggal_lahir'] : null;
                $data['jenis_kelamin'] = !empty($_POST['jenis_kelamin']) ? $_POST['jenis_kelamin'] : null;
                $data['alamat'] = isset($_POST['alamat']) ? trim($_POST['alamat']) : null;
                $data['telepon'] = isset($_POST['telepon']) ? trim($_POST['telepon']) : null;
                $data['class_id'] = !empty($_POST['class_id']) ? $_POST['class_id'] : null;
                
                // Handle Photo Upload
                if (!empty($_FILES['foto']['name'])) {
                    $foto_filename = $this->uploadFoto($_FILES['foto']);
                    if ($foto_filename) {
                        $data['foto'] = $foto_filename;
                    }
                }
            } else {
                // Guru or Admin - Use basic username
                $data['username'] = trim($_POST['username_basic']);
                $data['name'] = trim($_POST['name_basic']);
            }

            if ($this->userModel->addUser($data)) {
                 set_flash('Pengguna berhasil ditambahkan', 'success');
            } else {
                 set_flash('Gagal menambahkan pengguna (Username/NISN mungkin sudah ada)', 'danger');
            }
        }
        redirect('admin/users');
    }
    
    // Helper: Upload student photo
    private function uploadFoto($file) {
        $target_dir = "uploads/students/";
        if (!file_exists($target_dir)) {
            mkdir($target_dir, 0777, true);
        }
        
        // Validate file type
        $allowed_types = ['image/jpeg', 'image/png', 'image/jpg'];
        if (!in_array($file['type'], $allowed_types)) {
            set_flash('Foto harus format JPG atau PNG', 'warning');
            return null;
        }
        
        // Validate file size (max 2MB)
        if ($file['size'] > 2 * 1024 * 1024) {
            set_flash('Ukuran foto maksimal 2MB', 'warning');
            return null;
        }
        
        $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $filename = time() . '_' . uniqid() . '.' . $extension;
        $target_file = $target_dir . $filename;
        
        if (move_uploaded_file($file['tmp_name'], $target_file)) {
            return $filename;
        }
        
        return null;
    }

    public function profile() {
        $user = $this->userModel->getUserById($_SESSION['user_id']);
        $data = [
            'title' => 'Profil Saya',
            'active_menu' => 'profile',
            'user' => $user
        ];
        $this->view('layouts/header', $data);
        $this->view('profile', $data);
        $this->view('layouts/footer');
    }

    public function update_profile() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            // Check ID matches Session (Security)
            if($_POST['id'] != $_SESSION['user_id']) {
                die("Unauthorized");
            }

            $password = trim($_POST['password']);
            $password_confirm = trim($_POST['password_confirm']);
            
            if(!empty($password) && $password !== $password_confirm) {
                set_flash('Konfirmasi password tidak cocok', 'danger');
                redirect('admin/profile');
                return;
            }

            $data = [
                'id' => $_SESSION['user_id'],
                'name' => trim($_POST['name']),
                'username' => trim($_POST['username']),
                'role' => $_POST['role'], // Keep role
                'class_id' => $_POST['class_id'], // Keep class
                'password' => $password
            ];

            if ($this->userModel->updateUser($data)) {
                 $_SESSION['name'] = $data['name']; // Update session name
                 set_flash('Profil berhasil diperbarui', 'success');
            } else {
                 set_flash('Gagal memperbarui profil', 'danger');
            }
        }
        redirect('admin/profile');
    }

    public function update_user() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $role = trim($_POST['role']);
            $data = [
                'id' => $_POST['id'],
                'name' => trim($_POST['name']),
                'username' => trim($_POST['username']),
                'password' => trim($_POST['password']), // Optional
                'role' => $role,
                'class_id' => ($role == 'siswa') ? $_POST['class_id'] : null,
                
                // Biodata fields (mostly for siswa)
                'nisn' => ($role == 'siswa') ? trim($_POST['username']) : null, // Assuming NISN is Username
                'tempat_lahir' => isset($_POST['tempat_lahir']) ? trim($_POST['tempat_lahir']) : null,
                'tanggal_lahir' => !empty($_POST['tanggal_lahir']) ? $_POST['tanggal_lahir'] : null,
                'jenis_kelamin' => !empty($_POST['jenis_kelamin']) ? $_POST['jenis_kelamin'] : null,
                'alamat' => isset($_POST['alamat']) ? trim($_POST['alamat']) : null,
                'telepon' => isset($_POST['telepon']) ? trim($_POST['telepon']) : null,
                'foto' => null
            ];
            
            // Handle Photo Upload
            if (!empty($_FILES['foto']['name'])) {
                $foto_filename = $this->uploadFoto($_FILES['foto']);
                if ($foto_filename) {
                    $data['foto'] = $foto_filename; // Only set if upload success
                }
            }

            if ($this->userModel->updateUser($data)) {
                 set_flash('Pengguna berhasil diperbarui', 'success');
            } else {
                 set_flash('Gagal memperbarui pengguna', 'danger');
            }
        }
        redirect('admin/users');
    }

    // AJAX for Edit Modal
    public function get_user_json($id) {
        $user = $this->userModel->getUserById($id);
        echo json_encode($user);
    }

    public function reset_exam($user_id) {
         // This actually needs to look for BANNED or ONGOING sessions for this user and reset them?
         // Requirement says "reset admin tanpa kehilangan jawaban".
         // This implies unbanning the user so they can continue.
         
         // Assuming we want to unban the user's exam participation
         // We might need a separate page to see active sessions/bans or just do it from user list?
         // Let's assume we pass a session_id or we find the latest banned session for the user.
         
         // For simplicity, let's reset ALL banned sessions for this user to 'ongoing'
         if($this->userModel->resetBannedSessions($user_id)) {
              set_flash('User exam session has been unbanned/reset.', 'success');
         } else {
              set_flash('No banned sessions found or error occurred.', 'warning');
         }
         redirect('admin/users');
    }

    public function activity_logs() {
        $logs = $this->userModel->getLogs();
        $data = [
            'title' => 'Log Aktivitas Siswa',
            'active_menu' => 'logs',
            'logs' => $logs
        ];
        $this->view('layouts/header', $data);
        $this->view('admin/logs', $data);
        $this->view('layouts/footer');
    }
    
    // Print Exam Cards - Step 1: Select Class
    public function print_cards() {
        $classes = $this->classModel->getAllClasses();
        $data = [
            'title' => 'Cetak Kartu Ujian',
            'active_menu' => 'print_cards',
            'classes' => $classes
        ];
        $this->view('layouts/header', $data);
        $this->view('admin/print_cards', $data);
        $this->view('layouts/footer');
    }
    
    // Print Exam Cards - Step 2: Generate Cards for Selected Class
    public function generate_cards() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $class_id = $_POST['class_id'];
            
            // Get class info
            $class = $this->classModel->getClassById($class_id);
            
            // Get all students in this class
            $students = $this->userModel->getStudentsByClass($class_id);
            
            // Render printable card view (no header/footer for clean print)
            $data = [
                'class' => $class,
                'students' => $students
            ];
            $this->view('admin/print_cards_output', $data);
        } else {
            redirect('admin/print_cards');
        }
    }
    
    // Download Exam Cards as PDF
    public function download_cards_pdf() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $class_id = $_POST['class_id'];
            
            // Get class info
            $class = $this->classModel->getClassById($class_id);
            
            // Get all students in this class
            $students = $this->userModel->getStudentsByClass($class_id);
            
            // Clean any previous output
            if (ob_get_level()) {
                ob_end_clean();
            }
            
            // Suppress deprecated warnings from Dompdf (PHP 8.2+ compatibility)
            $old_error_level = error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
            
            // Initialize Dompdf first
            $vendorPath = dirname(dirname(dirname(__FILE__))) . '/vendor/autoload.php';
            require_once $vendorPath;
            
            $options = new \Dompdf\Options();
            $options->set('isRemoteEnabled', false); // Disable remote resources for faster processing
            $options->set('defaultFont', 'DejaVu Sans');
            $options->set('isHtml5ParserEnabled', true);
            
            $dompdf = new \Dompdf\Dompdf($options);
            
            // Generate HTML directly (not using view file to avoid extra output)
            $html = $this->generateCardHTML($class, $students);
            
            // Load HTML
            $dompdf->loadHtml($html);
            
            // Set paper size (A4 portrait)
            $dompdf->setPaper('A4', 'portrait');
            
            // Render PDF
            $dompdf->render();
            
            // Restore error reporting
            error_reporting($old_error_level);
            
            // Output PDF for download
            $filename = 'Kartu_Ujian_' . str_replace(' ', '_', $class->name) . '_' . date('Y-m-d') . '.pdf';
            $dompdf->stream($filename, ['Attachment' => true]);
            exit;
            
        } else {
            redirect('admin/print_cards');
        }
    }
    
    // Generate HTML for exam cards
private function generateCardHTML($class, $students) {
    $html = '<!DOCTYPE html>
<html lang="id">
<head>
<meta charset="UTF-8">
<title>Kartu Ujian - ' . htmlspecialchars($class->name) . '</title>
<style>
    * {
        margin: 0;
        padding: 0;
        box-sizing: border-box;
    }
    
    @page {
        size: A4 portrait;
        margin: 10mm;
    }
    
    body {
        font-family: DejaVu Sans, Arial, sans-serif;
        margin: 0;
        padding: 0;
    }
    
    .page-container {
        width: 100%;
    }
    
    .card-row {
        display: table;
        width: 100%;
        margin-bottom: 4mm;
        page-break-inside: avoid;
    }
    
    .card-wrapper {
        display: table-cell;
        width: 50%;
        padding: 2mm;
        vertical-align: top;
    }
    
    .card-container {
        width: 100%;
        height: 54mm;
        border: 2px solid #333;
        border-radius: 8px;
        padding: 8px;
        position: relative;
        background: white;
    }
    
    .card-header-bar {
        background: #667eea;
        color: white;
        padding: 4px 8px;
        margin: -8px -8px 6px -8px;
        text-align: center;
        font-size: 11px;
        font-weight: bold;
        border-radius: 6px 6px 0 0;
    }
    
    .card-body-content {
        display: table;
        width: 100%;
    }
    
    .photo-section {
        display: table-cell;
        width: 28mm;
        vertical-align: top;
    }
    
    .student-photo {
        width: 26mm;
        height: 35mm;
        border: 2px solid #ddd;
        background: #f8f9fa;
        text-align: center;
        padding-top: 14mm;
        font-size: 9px;
        color: #999;
    }
    
    .info-section {
        display: table-cell;
        padding-left: 5px;
        font-size: 8.5px;
        vertical-align: top;
    }
    
    .info-row {
        margin-bottom: 1.5px;
        line-height: 1.3;
    }
    
    .info-label {
        font-weight: bold;
        display: inline-block;
        width: 45px;
    }
    
    .signature-box {
        border: 1px solid #ddd;
        height: 16px;
        background: white;
        padding: 2px;
        font-size: 7px;
        color: #999;
        margin-top: 6px;
    }
    
    .card-footer-bar {
        position: absolute;
        bottom: 8px;
        left: 8px;
        right: 8px;
        text-align: center;
        font-size: 6.5px;
        color: #666;
        border-top: 1px solid #ddd;
        padding-top: 3px;
    }
    
    /* Page break after every 8 cards (4 rows) */
    .card-row:nth-child(4n) {
        page-break-after: always;
    }
</style>
</head>
<body>
<div class="page-container">';

    if (empty($students)) {
        $html .= '<div style="text-align:center; margin-top: 50px;">Tidak ada siswa di kelas ini.</div>';
    } else {
        $cardCount = 0;
        $totalStudents = count($students);
        
        foreach ($students as $index => $student) {
            // Start a new row every 2 cards
            if ($cardCount % 2 == 0) {
                $html .= '<div class="card-row">';
            }
            
            $ttl = '';
            if (!empty($student->tempat_lahir) && !empty($student->tanggal_lahir)) {
                $ttl = htmlspecialchars($student->tempat_lahir) . ', ' . date('d-m-Y', strtotime($student->tanggal_lahir));
            } else {
                $ttl = '-';
            }
            
            $html .= '<div class="card-wrapper">
                <div class="card-container">
                    <div class="card-header-bar">KARTU PESERTA UJIAN</div>
                    <div class="card-body-content">
                        <div class="photo-section">
                            <div class="student-photo">FOTO<br>SISWA</div>
                        </div>
                        <div class="info-section">
                            <div class="info-row">
                                <span class="info-label">NISN</span>
                                <span>: ' . htmlspecialchars($student->nisn ?? $student->username) . '</span>
                            </div>
                            <div class="info-row">
                                <span class="info-label">Nama</span>
                                <span>: <strong>' . htmlspecialchars(strtoupper($student->name)) . '</strong></span>
                            </div>
                            <div class="info-row">
                                <span class="info-label">Kelas</span>
                                <span>: ' . htmlspecialchars($class->name) . '</span>
                            </div>
                            <div class="info-row">
                                <span class="info-label">TTL</span>
                                <span>: ' . $ttl . '</span>
                            </div>
                            <div class="info-row">
                                <span class="info-label">L/P</span>
                                <span>: ' . htmlspecialchars($student->jenis_kelamin ?? '-') . '</span>
                            </div>
                            <div class="signature-box">
                                Tanda Tangan:
                            </div>
                        </div>
                    </div>
                    <div class="card-footer-bar">
                        Kartu ini harus dibawa saat ujian | Tahun Ajaran ' . date('Y') . '/' . (date('Y') + 1) . '
                    </div>
                </div>
            </div>';
            
            $cardCount++;
            
            // Close row after 2 cards OR if it's the last student with odd count
            if ($cardCount % 2 == 0 || $index == $totalStudents - 1) {
                // If last card is odd, add empty card wrapper to maintain layout
                if ($index == $totalStudents - 1 && $cardCount % 2 == 1) {
                    $html .= '<div class="card-wrapper"></div>';
                }
                $html .= '</div>'; // Close card-row
            }
        }
    }

    $html .= '</div>
</body></html>';
    
    return $html;
}

    public function delete_user($id) {
        if($this->userModel->deleteUser($id)) {
            set_flash('Pengguna berhasil dihapus', 'success');
        }
        redirect('admin/users');
    }
    
    // School Settings Management
    public function school_settings() {
        $settingsModel = $this->model('SchoolSettings');
        $settings = $settingsModel->getAllSettings();
        
        $data = [
            'title' => 'Pengaturan Sekolah',
            'active_menu' => 'school_settings',
            'school_name' => $settings['school_name'] ?? 'Sistem Ujian Online',
            'school_logo' => $settings['school_logo'] ?? null
        ];
        
        $this->view('layouts/header', $data);
        $this->view('admin/school_settings', $data);
        $this->view('layouts/footer');
    }
    
    public function update_school_settings() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $settingsModel = $this->model('SchoolSettings');
            
            // Update school name
            $school_name = trim($_POST['school_name']);
            if (!empty($school_name)) {
                $settingsModel->updateSetting('school_name', $school_name);
            }
            
            // Handle logo upload
            if (!empty($_FILES['school_logo']['name'])) {
                $logo_filename = $this->uploadSchoolLogo($_FILES['school_logo']);
                if ($logo_filename) {
                    // Delete old logo if exists
                    $old_logo = $settingsModel->getSetting('school_logo');
                    if ($old_logo && file_exists("uploads/school/" . $old_logo)) {
                        unlink("uploads/school/" . $old_logo);
                    }
                    $settingsModel->updateSetting('school_logo', $logo_filename);
                }
            }
            
            set_flash('Pengaturan sekolah berhasil diperbarui', 'success');
        }
        redirect('admin/school_settings');
    }
    
    private function uploadSchoolLogo($file) {
        $target_dir = "uploads/school/";
        if (!file_exists($target_dir)) {
            mkdir($target_dir, 0777, true);
        }
        
        // Validate file type
        $allowed_types = ['image/jpeg', 'image/png', 'image/jpg'];
        if (!in_array($file['type'], $allowed_types)) {
            set_flash('Logo harus format JPG atau PNG', 'warning');
            return null;
        }
        
        // Validate file size (max 2MB)
        if ($file['size'] > 2 * 1024 * 1024) {
            set_flash('Ukuran logo maksimal 2MB', 'warning');
            return null;
        }
        
        $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $filename = 'logo_' . time() . '.' . $extension;
        $target_file = $target_dir . $filename;
        
        if (move_uploaded_file($file['tmp_name'], $target_file)) {
            return $filename;
        }
        
        set_flash('Gagal mengupload logo', 'danger');
        return null;
    }

    // Server Management
    // Server Management
    public function server() {
        $settingsModel = $this->model('SchoolSettings');
        
        $active_users_count = $this->userModel->countActiveUsers(5); // For KPI card
        $active_users_list = $this->userModel->getActiveUsersList(5); // For List Table
        
        $settings = $settingsModel->getAllSettings();
        
        $data = [
            'title' => 'Server & Traffic Monitor',
            'active_menu' => 'server',
            'active_users' => $active_users_count, 
            'active_users_list' => $active_users_list,
            'max_users' => $settings['max_concurrent_users'] ?? 1000,
            'maintenance_mode' => $settings['maintenance_mode'] ?? '0',
            'server_message' => $settings['server_message'] ?? ''
        ];
        
        $this->view('layouts/header', $data);
        $this->view('admin/server_monitor', $data);
        $this->view('layouts/footer');
    }

    public function update_server_config() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $settingsModel = $this->model('SchoolSettings');
            
            $max_users = $_POST['max_concurrent_users'];
            $maintenance = isset($_POST['maintenance_mode']) ? '1' : '0';
            $message = $_POST['server_message'];
            
            $settingsModel->updateSetting('max_concurrent_users', $max_users);
            $settingsModel->updateSetting('maintenance_mode', $maintenance);
            $settingsModel->updateSetting('server_message', $message);
            
            set_flash('Konfigurasi server berhasil diperbarui', 'success');
        }
        redirect('admin/server');
    }

    public function suspend_user() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $user_id = $_POST['user_id'];
            $minutes = $_POST['minutes']; // 5, 15, 60
            
            if($this->userModel->suspendUser($user_id, $minutes)) {
                set_flash("User berhasil disuspend selama $minutes menit.", 'warning');
            } else {
                set_flash('Gagal mensuspend user.', 'danger');
            }
        }
        redirect('admin/server');
    }

    public function unsuspend_user() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $user_id = $_POST['user_id'];
            
            if($this->userModel->unsuspendUser($user_id)) {
                set_flash("User berhasil diaktifkan kembali.", 'success');
            } else {
                set_flash('Gagal mengaktifkan user.', 'danger');
            }
        }
        redirect('admin/server');
    }
}
