<?php
// Load Composer autoloader with absolute path
$autoloadPath = __DIR__ . '/../../vendor/autoload.php';
if (file_exists($autoloadPath)) {
    require_once $autoloadPath;
}

use Dompdf\Dompdf;
use Dompdf\Options;

class GuruController extends Controller {
    private $examModel;
    private $guruProfile;
    private $classModel;
    private $userModel;
    private $subjectModel;

    public function __construct() {
        AuthMiddleware::check(['guru']);
        $this->examModel = $this->model('Exam');
        $this->classModel = $this->model('ClassModel');
        $this->userModel = $this->model('User'); 
        $this->subjectModel = $this->model('SubjectModel'); // Load Subject Model
        
        // Secure: Get Guru ID from User ID
        $this->guruProfile = $this->examModel->getGuruProfileByUserId($_SESSION['user_id']);
        if(!$this->guruProfile) {
             die("Guru Profile not found. Please contact admin.");
        }
    }

    // Helper to ensure Exam belongs to current Guru
    private function checkOwnership($exam_id) {
        $exam = $this->examModel->getExamById($exam_id);
        if (!$exam || $exam->guru_id != $this->guruProfile->id) {
            set_flash('Unauthorized Access', 'danger');
            redirect('guru/dashboard');
            exit;
        }
        return $exam;
    }

    public function profile() {
        $user = $this->userModel->getUserById($_SESSION['user_id']);
        $data = [
            'title' => 'Profil Saya',
            'active_menu' => 'profile',
            'user' => $user
        ];
        $this->view('layouts/header', $data);
        $this->view('profile', $data);
        $this->view('layouts/footer');
    }

    public function update_profile() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            if($_POST['id'] != $_SESSION['user_id']) die("Unauthorized");

            $password = trim($_POST['password']);
            $password_confirm = trim($_POST['password_confirm']);
            
            if(!empty($password) && $password !== $password_confirm) {
                set_flash('Konfirmasi password tidak cocok', 'danger');
                redirect('guru/profile');
                return;
            }

            $data = [
                'id' => $_SESSION['user_id'],
                'name' => trim($_POST['name']),
                'username' => trim($_POST['username']),
                'role' => $_POST['role'], 
                'class_id' => $_POST['class_id'],
                'password' => $password
            ];

            if ($this->userModel->updateUser($data)) {
                 $_SESSION['name'] = $data['name'];
                 set_flash('Profil berhasil diperbarui', 'success');
            } else {
                 set_flash('Gagal memperbarui profil', 'danger');
            }
        }
        redirect('guru/profile');
    }

    public function bank() {
        // Feature: "Bank Soal per Guru"
        // List all questions owned by this guru, grouped by Mapel
        $questions = $this->examModel->getAllQuestionsByGuru($this->guruProfile->id);
        
        $data = [
            'title' => 'Bank Soal Saya',
            'active_menu' => 'bank_soal',
            'questions' => $questions,
            'active_menu' => 'bank' // Correct active menu for sidebar
        ];
        $this->view('layouts/header', $data);
        $this->view('guru/bank_soal', $data);
        $this->view('layouts/footer');
    }

    public function dashboard() {
        // Fetch Stats
        $exams = $this->examModel->getExamsByGuru($_SESSION['user_id']);
        $questions = $this->examModel->getAllQuestionsByGuru($this->guruProfile->id);
        
        $stats = [
            'total_exams' => count($exams),
            'total_questions' => count($questions),
            'published_exams' => 0 // Calculate below
        ];

        foreach($exams as $ex) {
            if($ex->status == 'published') $stats['published_exams']++;
        }

        $data = [
            'title' => 'Guru Dashboard',
            'active_menu' => 'dashboard',
            'stats' => $stats
        ];
        $this->view('layouts/header', $data);
        $this->view('guru/dashboard', $data);
        $this->view('layouts/footer');
    }

    public function exams() {
        $exams = $this->examModel->getExamsByGuru($_SESSION['user_id']);
        $data = [
            'title' => 'Manage Exams',
            'active_menu' => 'exams',
            'exams' => $exams
        ];
        $this->view('layouts/header', $data);
        $this->view('guru/exam_index', $data);
        $this->view('layouts/footer');
    }

    public function create_exam() {
        $classes = $this->classModel->getAllClasses(); 
        $subjects = $this->subjectModel->getAllSubjects(); // Get subjects
        $data = [
            'title' => 'Create New Exam',
            'active_menu' => 'exams',
            'classes' => $classes,
            'subjects' => $subjects
        ];
        $this->view('layouts/header', $data);
        $this->view('guru/exam_create', $data);
        $this->view('layouts/footer');
    }

    public function store_exam() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $data = [
                'guru_id' => $_SESSION['user_id'],
                'title' => trim($_POST['title']),
                'description' => trim($_POST['description']),
                'duration_minutes' => trim($_POST['duration_minutes']),
                'start_datetime' => trim($_POST['start_datetime']),
                'end_datetime' => trim($_POST['end_datetime']),
                'token' => trim($_POST['token']),
                'status' => 'draft', 
                'class_id' => trim($_POST['class_id']),
                'subject_id' => trim($_POST['subject_id']) // Add subject_id
            ];

            if ($this->examModel->addExam($data)) {
                set_flash('Ujian berhasil dibuat!', 'success');
                redirect('guru/exams');
            } else {
                die('Terjadi kesalahan');
            }
        } else {
            redirect('guru/exams');
        }
    }

    public function delete_exam($id) {
        if ($this->examModel->deleteExam($id)) {
            set_flash('Ujian berhasil dihapus', 'success');
        } else {
            set_flash('Terjadi kesalahan', 'danger');
        }
        redirect('guru/exams');
    }

    public function edit_exam($id) {
        $exam = $this->checkOwnership($id);
        $classes = $this->classModel->getAllClasses();
        $subjects = $this->subjectModel->getAllSubjects();
        
        $data = [
            'title' => 'Edit Ujian',
            'active_menu' => 'exams',
            'exam' => $exam,
            'classes' => $classes,
            'subjects' => $subjects
        ];
        
        $this->view('layouts/header', $data);
        $this->view('guru/exam_edit', $data);
        $this->view('layouts/footer');
    }

    public function update_exam() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $id = $_POST['id'];
            $this->checkOwnership($id);

            $data = [
                'id' => $id,
                'guru_id' => $_SESSION['user_id'],
                'title' => trim($_POST['title']),
                'description' => trim($_POST['description']),
                'duration_minutes' => trim($_POST['duration_minutes']),
                'start_datetime' => trim($_POST['start_datetime']),
                'end_datetime' => trim($_POST['end_datetime']),
                'token' => trim($_POST['token']),
                'class_id' => trim($_POST['class_id']),
                'subject_id' => trim($_POST['subject_id'])
            ];

            if ($this->examModel->updateExam($data)) {
                set_flash('Ujian berhasil diperbarui', 'success');
                redirect('guru/exams');
            } else {
                set_flash('Gagal memperbarui ujian', 'danger');
                redirect('guru/edit_exam/' . $id);
            }
        }
    }

    public function set_status($id, $status) {
        $this->checkOwnership($id);
        
        // Validate Status
        if (!in_array($status, ['published', 'closed', 'draft'])) {
             set_flash('Status tidak valid', 'danger');
             redirect('guru/exams');
             exit;
        }

        if ($this->examModel->updateStatus($id, $status)) {
            $msg = ($status == 'published') ? 'Ujian diaktifkan (Published)' : 'Ujian dinonaktifkan (Closed)';
            set_flash($msg, 'success');
        } else {
            set_flash('Gagal mengubah status', 'danger');
        }
        redirect('guru/exams');
    }

    /* --- Question Management --- */

    public function questions($exam_id) {
        $exam = $this->examModel->getExamById($exam_id);
        $questions = $this->examModel->getQuestionsByExamId($exam_id);
        
        $data = [
            'title' => 'Manage Questions - ' . $exam->title,
            'active_menu' => 'exams', // Keep exams active
            'exam' => $exam,
            'questions' => $questions
        ];

        $this->view('layouts/header', $data);
        $this->view('guru/question_index', $data);
        $this->view('layouts/footer');
    }

    public function add_question($exam_id) {
        $exam = $this->examModel->getExamById($exam_id);
        $data = [
            'title' => 'Add Question - ' . $exam->title,
            'active_menu' => 'exams',
            'exam' => $exam
        ];
        $this->view('layouts/header', $data);
        $this->view('guru/question_create', $data);
        $this->view('layouts/footer');
    }

    public function store_question() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $exam_id = $_POST['exam_id'];
            $type = isset($_POST['type']) ? $_POST['type'] : 'pilihan_ganda';
            
            $data = [
                'exam_id' => $exam_id,
                'question_text' => trim($_POST['question_text']),
                'type' => $type,
                'option_a' => null, 'option_b' => null, 'option_c' => null, 'option_d' => null, 'option_e' => null,
                'option_a_image' => null, 'option_b_image' => null, 'option_c_image' => null, 
                'option_d_image' => null, 'option_e_image' => null,
                'answer_key' => isset($_POST['answer_key']) ? $_POST['answer_key'] : null
            ];

            // Handle Question Image
            if (!empty($_FILES['question_image']['name'])) {
                $data['question_image'] = $this->uploadFile($_FILES['question_image']);
            }

            // Handle Types
            if ($type == 'pilihan_ganda') {
                $data['option_a'] = trim($_POST['option_a']);
                $data['option_b'] = trim($_POST['option_b']);
                $data['option_c'] = trim($_POST['option_c']);
                $data['option_d'] = trim($_POST['option_d']);
                $data['option_e'] = trim($_POST['option_e']);
                
                // Handle Option Images for standard multiple choice (A-E)
                $options = ['a', 'b', 'c', 'd', 'e'];
                foreach ($options as $opt) {
                    $fileKey = 'option_' . $opt . '_image';
                    if (!empty($_FILES[$fileKey]['name'])) {
                        $data['option_' . $opt . '_image'] = $this->uploadFile($_FILES[$fileKey]);
                    }
                }
            } 
            elseif ($type == 'benar_salah') {
                // Handle True/False type - use advanced options for better validation
                $data['advanced_options'] = [];
                if (isset($_POST['adv_options']) && is_array($_POST['adv_options'])) {
                    foreach ($_POST['adv_options'] as $key => $opt) {
                        $optData = [
                            'text' => $opt['text'],
                            'pair_text' => null,
                            'is_correct' => isset($opt['is_correct']) ? 1 : 0
                        ];
                        $data['advanced_options'][] = $optData;
                        
                        // Set answer_key based on which option is marked correct
                        if ($optData['is_correct'] == 1) {
                            $data['answer_key'] = $opt['text']; // 'Benar' or 'Salah'
                        }
                    }
                }
            }
            elseif ($type == 'pilihan_ganda_kompleks' || $type == 'menjodohkan') {
                // Process Advanced Options Array
                // Expected Input: options[index][text], options[index][file], options[index][is_correct], options[index][pair]
                $data['advanced_options'] = [];
                if (isset($_POST['adv_options']) && is_array($_POST['adv_options'])) {
                    foreach ($_POST['adv_options'] as $key => $opt) {
                        $optData = [
                            'text' => $opt['text'],
                            'pair_text' => isset($opt['pair']) ? $opt['pair'] : null,
                            'is_correct' => isset($opt['is_correct']) ? 1 : 0
                        ];
                        
                        // Handle Option Image Upload
                        $fileKey = "adv_opt_img_" . $key;
                        if (!empty($_FILES[$fileKey]['name'])) {
                             $optData['image'] = $this->uploadFile($_FILES[$fileKey]);
                        }
                        
                        $data['advanced_options'][] = $optData;
                    }
                }
            }
            elseif ($type == 'isian_singkat') {
                // Handle short answer key
                if (isset($_POST['answer_key_text'])) {
                    $data['answer_key'] = trim($_POST['answer_key_text']);
                }
            }
            elseif ($type == 'uraian') {
                // Essay type - no specific answer key needed
                $data['answer_key'] = '-'; // Placeholder
            }

            if ($this->examModel->addQuestion($data)) {
                set_flash('Soal berhasil ditambahkan', 'success');
                redirect('guru/questions/' . $exam_id);
            } else {
                die('Terjadi kesalahan');
            }
        }
    }


    private function uploadFile($file) {
        $target_dir = "uploads/questions/"; // Make sure this exists
        if (!file_exists($target_dir)) {
            mkdir($target_dir, 0777, true);
        }
        
        $filename = time() . '_' . basename($file["name"]);
        $target_file = $target_dir . $filename;
        
        // Simple check
        $check = getimagesize($file["tmp_name"]);
        if($check !== false) {
             if (move_uploaded_file($file["tmp_name"], $target_file)) {
                 return $filename; // Store relative filename or path
             }
        }
        return null; // Fail
    }

    public function delete_question($id, $exam_id) {
        if ($this->examModel->deleteQuestion($id)) {
            set_flash('Soal berhasil dihapus', 'success');
        } else {
            set_flash('Gagal menghapus soal', 'danger');
        }
        redirect('guru/questions/' . $exam_id);
    }
    
    public function grading($exam_id) {
        $this->checkOwnership($exam_id);
        // Load participants and answers...
        // For now just view placeholder
        $data = [
            'title' => 'Input Nilai Essay',
            'active_menu' => 'exams'
        ];
        $this->view('layouts/header', $data);
        $this->view('guru/grading', $data);
        $this->view('layouts/footer');
    }

    public function monitor($exam_id) {
        $exam = $this->checkOwnership($exam_id);
        $participants = $this->examModel->getExamResults($exam_id);
        
        $data = [
            'title' => 'Monitoring Ujian: ' . $exam->title,
            'active_menu' => 'exams',
            'exam' => $exam,
            'participants' => $participants
        ];
        $this->view('layouts/header', $data);
        $this->view('guru/exam_monitor', $data); // New view for realtime monitoring? Or reuse results?
        $this->view('layouts/footer');
    }

    public function results($exam_id) {
        $exam = $this->checkOwnership($exam_id);
        $results = $this->examModel->getExamResults($exam_id);
        // Stats removed for now as method missing
        
        $data = [
            'title' => 'Hasil Ujian: ' . $exam->title,
            'active_menu' => 'exams',
            'exam' => $exam,
            'results' => $results
        ];
        $this->view('layouts/header', $data);
        $this->view('guru/results', $data);
        $this->view('layouts/footer');
    }

    public function publish_results($exam_id) {
        $exam = $this->checkOwnership($exam_id);
        // Toggle
        $new_status = ($exam->show_results == 1) ? 0 : 1;
        
        $this->examModel->updateShowResults($exam_id, $new_status);
        
        set_flash('Status publikasi nilai berhasil diubah.', 'success');
        redirect('guru/results/' . $exam_id);
    }
    
    public function grade_student($session_id) {
        // Need to fetch session details implies joining with exam to check ownership
        $session = $this->examModel->getExamSession($session_id);
        if (!$session) die('Session not found');
        
        $exam = $this->checkOwnership($session->exam_id);
        
        $questions = $this->examModel->getQuestionsByExamId($session->exam_id);
        // Note: $session_id here is actually the exam_participants.id (participant_id)
        $answers = $this->examModel->getStudentAnswers($session_id);
        
        // Map Answers by question_id for easy lookup
        $ans_map = [];
        foreach($answers as $a) $ans_map[$a->question_id] = $a;

        $student = $this->userModel->getUserById($session->student_id);

        $data = [
            'title' => 'Koreksi Jawaban: ' . $student->name,
            'active_menu' => 'exams',
            'exam' => $exam,
            'questions' => $questions,
            'ans_map' => $ans_map,
            'session' => $session,
            'student' => $student
        ];
        
        $this->view('layouts/header', $data);
        $this->view('guru/grade_student', $data); // We will create this
        $this->view('layouts/footer');
    }

    public function save_grades($session_id) {
         if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['scores'])) {
             $session = $this->examModel->getExamSession($session_id);
             $exam = $this->checkOwnership($session->exam_id); // Security check
             
             $scores = $_POST['scores']; // [ans_id => score, ...]
             $total_earned = 0;
             $question_count = count($scores); // Be careful, count all questions or just graded ones?
             // Better re-fetch all questions to get max score.
             
             $questions = $this->examModel->getQuestionsByExamId($exam->id);
             $max_score = count($questions) * 10; 
             
             // Setup map for easier total calc
             // But $_POST['scores'] is keyed by Answer ID (or Question ID?). View should send carefully.
             // Let's assume $_POST['scores'][question_id] to be safe? 
             // Or $_POST['scores'][answer_id]? 
             // Logic: Update student_answers score.
             
             foreach ($scores as $ans_id => $val) {
                 $val = floatval($val);
                 if($val < 0) $val = 0;
                 if($val > 10) $val = 10; // Cap at max per question
                 
                 $this->examModel->updateAnswerScore($ans_id, $val);
                 $total_earned += $val;
             }
             
             // Normalize Final Grade
             $final_grade = ($max_score > 0) ? ($total_earned / $max_score) * 100 : 0;
             
             $this->examModel->finishExamSession($session_id, $final_grade);
             set_flash('Nilai berhasil disimpan. Total: '.$final_grade, 'success');
             redirect('guru/results/' . $exam->id);
         }
    }

    public function export_scores($exam_id, $type = 'excel') {
        $exam = $this->checkOwnership($exam_id);
        $results = $this->examModel->getExamResults($exam_id);
        
        if ($type == 'excel') {
            // Improved Excel Export - HTML-based but cleaner and more compatible
            header("Content-Type: application/vnd.ms-excel; charset=utf-8");
            header("Content-Disposition: attachment; filename=Laporan_Hasil_Ujian_" . preg_replace('/[^a-zA-Z0-9]/', '_', $exam->title) . "_" . date('Ymd_His') . ".xls");
            header("Pragma: no-cache");
            header("Expires: 0");
            
            // Start HTML with proper encoding
            echo "\xEF\xBB\xBF"; // UTF-8 BOM
            ?>
<!DOCTYPE html>
<html>
<head>
    <meta http-equiv="Content-Type" content="text/html; charset=utf-8">
    <style>
        body { font-family: Calibri, Arial, sans-serif; }
        table { border-collapse: collapse; width: 100%; }
        
        /* Header Styles */
        .main-title { 
            background-color: #4e73df; 
            color: #ffffff; 
            font-size: 20pt; 
            font-weight: bold; 
            padding: 15px; 
            text-align: center; 
            border: 2px solid #2e59d9; 
        }
        
        .school-name { 
            background-color: #667eea; 
            color: #ffffff; 
            font-size: 16pt; 
            font-weight: bold; 
            padding: 12px; 
            text-align: center; 
            border: 2px solid #5568d3; 
        }
        
        .section-header {
            background-color: #e3f2fd;
            color: #01579b;
            font-weight: bold;
            font-size: 13pt;
            padding: 10px;
            text-align: center;
            border: 1px solid #90caf9;
        }
        
        /* Info Section */
        .info-label { 
            background-color: #f5f5f5; 
            font-weight: bold; 
            padding: 8px; 
            border: 1px solid #cccccc; 
            font-size: 11pt; 
            width: 180px; 
        }
        
        .info-value { 
            background-color: #ffffff; 
            padding: 8px; 
            border: 1px solid #cccccc; 
            font-size: 11pt; 
        }
        
        /* Table Header */
        .table-header { 
            background-color: #4e73df; 
            color: #ffffff; 
            font-weight: bold; 
            padding: 10px 6px; 
            text-align: center; 
            border: 1px solid #2e59d9; 
            font-size: 11pt; 
        }
        
        /* Data Cells */
        .data-cell { 
            padding: 8px; 
            border: 1px solid #dddddd; 
            font-size: 10pt; 
        }
        
        .cell-center { 
            text-align: center; 
        }
        
        .cell-number { 
            text-align: center; 
            background-color: #f8f9fa; 
            font-weight: bold; 
        }
        
        /* Status Colors */
        .status-ongoing { 
            background-color: #fff3cd; 
            color: #856404; 
            font-weight: bold; 
            text-align: center; 
        }
        
        .status-completed { 
            background-color: #d4edda; 
            color: #155724; 
            font-weight: bold; 
            text-align: center; 
        }
        
        .status-banned { 
            background-color: #f8d7da; 
            color: #721c24; 
            font-weight: bold; 
            text-align: center; 
        }
        
        /* Score Cell */
        .score-cell { 
            background-color: #d4fc79; 
            font-weight: bold; 
            font-size: 12pt; 
            text-align: center; 
            border: 1px solid #4caf50; 
        }
        
        /* Statistics */
        .stats-header {
            background-color: #ffecd2;
            color: #e65100;
            font-weight: bold;
            font-size: 12pt;
            padding: 10px;
            text-align: center;
            border: 2px solid #ff9966;
        }
        
        .stats-box { 
            padding: 8px; 
            text-align: center; 
            font-size: 11pt; 
            font-weight: bold; 
            border: 1px solid #cccccc; 
        }
        
        .stats-completed { 
            background-color: #e8f5e9; 
            color: #2e7d32; 
        }
        
        .stats-ongoing { 
            background-color: #fff3e0; 
            color: #e65100; 
        }
        
        .stats-banned { 
            background-color: #ffebee; 
            color: #c62828; 
        }
        
        /* Summary Row */
        .summary-label { 
            background-color: #e3f2fd; 
            font-weight: bold; 
            text-align: right; 
            padding: 10px 15px; 
            border: 2px solid #90caf9; 
            font-size: 12pt; 
        }
        
        .summary-value { 
            background-color: #fff3cd; 
            font-weight: bold; 
            font-size: 14pt; 
            text-align: center; 
            padding: 10px; 
            border: 2px solid #ffc107; 
        }
        
        /* Zebra Striping */
        tr.odd { background-color: #ffffff; }
        tr.even { background-color: #f8f9fc; }
    </style>
</head>
<body>
    <table>
        <!-- Main Title -->
        <tr>
            <td colspan="7" class="main-title">LAPORAN HASIL UJIAN</td>
        </tr>
        
        <!-- School Name -->
        <tr>
            <td colspan="7" class="school-name"><?php echo strtoupper(APP_NAME); ?></td>
        </tr>
        
        <!-- Spacer -->
        <tr><td colspan="7" style="height: 10px;"></td></tr>
        
        <!-- Info Section Header -->
        <tr>
            <td colspan="7" class="section-header">INFORMASI UJIAN</td>
        </tr>
        
        <!-- Exam Details -->
        <tr>
            <td colspan="2" class="info-label">Nama Ujian</td>
            <td colspan="5" class="info-value"><?php echo htmlspecialchars($exam->title); ?></td>
        </tr>
        <tr>
            <td colspan="2" class="info-label">Deskripsi</td>
            <td colspan="5" class="info-value"><?php echo htmlspecialchars($exam->description); ?></td>
        </tr>
        <tr>
            <td colspan="2" class="info-label">Tanggal Export</td>
            <td colspan="5" class="info-value"><?php echo date('d F Y, H:i:s'); ?> WIB</td>
        </tr>
        
        <!-- Spacer -->
        <tr><td colspan="7" style="height: 15px;"></td></tr>
        
        <!-- Table Header -->
        <tr>
            <td class="table-header" width="5%">No</td>
            <td class="table-header" width="25%">Nama Siswa</td>
            <td class="table-header" width="12%">Kelas</td>
            <td class="table-header" width="16%">Waktu Mulai</td>
            <td class="table-header" width="16%">Waktu Selesai</td>
            <td class="table-header" width="14%">Status</td>
            <td class="table-header" width="12%">Nilai</td>
        </tr>
        
        <!-- Table Body -->
        <?php
        $no = 1;
        $total_score = 0;
        $count_completed = 0;
        $count_ongoing = 0;
        $count_banned = 0;
        
        foreach ($results as $row) {
            $score = isset($row->score) && $row->score !== null ? floatval($row->score) : 0;
            $status_class = 'status-' . strtolower($row->status);
            $row_class = ($no % 2 == 0) ? 'even' : 'odd';
            
            // Count statistics
            if ($row->status == 'completed') {
                $total_score += $score;
                $count_completed++;
            } elseif ($row->status == 'ongoing') {
                $count_ongoing++;
            } elseif ($row->status == 'banned') {
                $count_banned++;
            }
        ?>
        <tr class="<?php echo $row_class; ?>">
            <td class="data-cell cell-number"><?php echo $no++; ?></td>
            <td class="data-cell"><?php echo htmlspecialchars($row->student_name); ?></td>
            <td class="data-cell cell-center"><?php echo htmlspecialchars($row->class_name); ?></td>
            <td class="data-cell cell-center"><?php echo $row->start_time ?? '-'; ?></td>
            <td class="data-cell cell-center"><?php echo $row->end_time ?? '-'; ?></td>
            <td class="data-cell <?php echo $status_class; ?>"><?php echo strtoupper($row->status); ?></td>
            <td class="score-cell"><?php echo number_format($score, 2); ?></td>
        </tr>
        <?php } ?>
        
        <!-- Spacer -->
        <tr><td colspan="7" style="height: 10px;"></td></tr>
        
        <!-- Statistics Header -->
        <tr>
            <td colspan="7" class="stats-header">STATISTIK HASIL UJIAN</td>
        </tr>
        
        <!-- Statistics Data -->
        <tr>
            <td colspan="2" class="stats-box stats-completed">Selesai: <?php echo $count_completed; ?></td>
            <td colspan="2" class="stats-box stats-ongoing">Berlangsung: <?php echo $count_ongoing; ?></td>
            <td colspan="3" class="stats-box stats-banned">Diblokir: <?php echo $count_banned; ?></td>
        </tr>
        
        <!-- Average Score -->
        <?php $avg_score = $count_completed > 0 ? $total_score / $count_completed : 0; ?>
        <tr>
            <td colspan="6" class="summary-label">RATA-RATA NILAI (dari siswa yang selesai):</td>
            <td class="summary-value"><?php echo number_format($avg_score, 2); ?></td>
        </tr>
        
        <!-- Spacer -->
        <tr><td colspan="7" style="height: 10px;"></td></tr>
        
        <!-- Footer -->
        <tr>
            <td colspan="7" style="text-align: center; font-size: 9pt; color: #666; font-style: italic; padding: 8px;">
                Dokumen ini digenerate secara otomatis oleh Sistem E-Exam pada <?php echo date('d/m/Y H:i:s'); ?>
            </td>
        </tr>
    </table>
</body>
</html>
<?php
            exit;
        } elseif ($type == 'pdf') {
            // Professional PDF Export using Dompdf Library
            try {
                // Check if Dompdf classes are available
                if (!class_exists('Dompdf\Dompdf')) {
                    die('Error: Dompdf library not found. Please run: composer require dompdf/dompdf');
                }
                
                // Configure Dompdf
                $options = new Options();
                $options->set('isHtml5ParserEnabled', true);
                $options->set('isRemoteEnabled', true);
                $options->set('defaultFont', 'Arial');
                $options->set('debugPng', false);
                $options->set('debugKeepTemp', false);
                $options->set('debugCss', false);
                
                $dompdf = new Dompdf($options);
                
                // Build HTML content for PDF
                $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <style>
        @page {
            margin: 1cm 1.5cm;
        }
        body {
            font-family: Arial, sans-serif;
            font-size: 11pt;
            line-height: 1.4;
        }
        .header {
            text-align: center;
            border-bottom: 3px solid #2e59d9;
            padding-bottom: 15px;
            margin-bottom: 20px;
        }
        .header h1 {
            margin: 5px 0;
            font-size: 20pt;
            color: #2e59d9;
        }
        .header h2 {
            margin: 5px 0;
            font-size: 16pt;
            color: #555;
        }
        .header p {
            margin: 3px 0;
            font-size: 10pt;
            color: #666;
        }
        .info-box {
            background-color: #f8f9fc;
            border: 1px solid #dddddd;
            padding: 12px;
            margin-bottom: 20px;
            border-radius: 5px;
        }
        .info-box table {
            width: 100%;
            border: none;
        }
        .info-box td {
            padding: 5px;
            border: none;
        }
        .info-box td:first-child {
            width: 150px;
            font-weight: bold;
            color: #333;
        }
        table.data-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 10px;
        }
        table.data-table th {
            background-color: #4e73df;
            color: white;
            padding: 10px 8px;
            text-align: center;
            font-weight: bold;
            border: 1px solid #2e59d9;
            font-size: 10pt;
        }
        table.data-table td {
            padding: 8px;
            border: 1px solid #dddddd;
            font-size: 10pt;
        }
        table.data-table tr:nth-child(even) {
            background-color: #f8f9fc;
        }
        .text-center { text-align: center; }
        .text-right { text-align: right; }
        .text-bold { font-weight: bold; }
        .score-cell {
            background-color: #e7f4e4 !important;
            font-weight: bold;
            font-size: 11pt;
            text-align: center;
        }
        .status-ongoing { color: #f6c23e; font-weight: bold; }
        .status-completed { color: #1cc88a; font-weight: bold; }
        .status-banned { color: #e74a3b; font-weight: bold; }
        .summary-row {
            background-color: #e3f2fd !important;
            font-weight: bold;
        }
        .summary-score {
            background-color: #fff3cd !important;
            font-size: 13pt;
            font-weight: bold;
        }
    </style>
</head>
<body>';
                
                // Header
                $html .= '<div class="header">';
                $html .= '<h1>LAPORAN HASIL UJIAN</h1>';
                $html .= '<h2>' . strtoupper(APP_NAME) . '</h2>';
                $html .= '<p>Jl. Contoh No. 123, Kota | Telp: (021) 12345678</p>';
                $html .= '</div>';
                
                // Info Box
                $html .= '<div class="info-box">';
                $html .= '<table>';
                $html .= '<tr><td>Nama Ujian</td><td>: ' . htmlspecialchars($exam->title) . '</td></tr>';
                $html .= '<tr><td>Deskripsi</td><td>: ' . htmlspecialchars($exam->description) . '</td></tr>';
                $html .= '<tr><td>Tanggal Cetak</td><td>: ' . date('d F Y, H:i:s') . ' WIB</td></tr>';
                $html .= '</table>';
                $html .= '</div>';
                
                // Data Table
                $html .= '<table class="data-table">';
                $html .= '<thead>';
                $html .= '<tr>';
                $html .= '<th width="5%">No</th>';
                $html .= '<th width="30%">Nama Siswa</th>';
                $html .= '<th width="15%">Kelas</th>';
                $html .= '<th width="20%">Waktu Mulai</th>';
                $html .= '<th width="12%">Status</th>';
                $html .= '<th width="12%">Nilai</th>';
                $html .= '</tr>';
                $html .= '</thead>';
                $html .= '<tbody>';
                
                if(empty($results)) {
                    $html .= '<tr><td colspan="6" class="text-center">Belum ada data hasil ujian</td></tr>';
                } else {
                    $no = 1;
                    $total_score = 0;
                    $count_completed = 0;
                    
                    foreach($results as $row) {
                        $score = isset($row->score) && $row->score !== null ? floatval($row->score) : 0;
                        $status_class = 'status-' . strtolower($row->status);
                        
                        if ($row->status == 'completed') {
                            $total_score += $score;
                            $count_completed++;
                        }
                        
                        $html .= '<tr>';
                        $html .= '<td class="text-center">' . $no++ . '</td>';
                        $html .= '<td>' . htmlspecialchars($row->student_name) . '</td>';
                        $html .= '<td>' . htmlspecialchars($row->class_name) . '</td>';
                        $html .= '<td class="text-center">' . date('d-m-Y H:i', strtotime($row->start_time ?? 'now')) . '</td>';
                        $html .= '<td class="text-center ' . $status_class . '">' . ucfirst($row->status) . '</td>';
                        $html .= '<td class="score-cell">' . number_format($score, 2) . '</td>';
                        $html .= '</tr>';
                    }
                    
                    // Summary Row
                    $avg_score = $count_completed > 0 ? $total_score / $count_completed : 0;
                    $html .= '<tr class="summary-row">';
                    $html .= '<td colspan="5" class="text-right" style="padding-right: 15px;">Rata-rata Nilai:</td>';
                    $html .= '<td class="summary-score text-center">' . number_format($avg_score, 2) . '</td>';
                    $html .= '</tr>';
                }
                
                $html .= '</tbody>';
                $html .= '</table>';
                
                // Footer info
                $html .= '<div style="margin-top: 30px; font-size: 9pt; color: #666;">';
                $html .= '<p><em>Dokumen ini digenerate otomatis oleh sistem E-Exam Sekolah</em></p>';
                $html .= '</div>';
                
                $html .= '</body></html>';
                
                // Load HTML to Dompdf
                $dompdf->loadHtml($html);
                
                // Setup paper size and orientation
                $dompdf->setPaper('A4', 'portrait');
                
                // Render PDF
                $dompdf->render();
                
                // Output PDF to browser - Direct Download (no print dialog)
                $filename = 'Laporan_Ujian_' . preg_replace('/[^a-zA-Z0-9]/', '_', $exam->title) . '_' . date('Ymd_His') . '.pdf';
                
                // Stream directly to download (0 = inline/download, 1 = attachment)
                $dompdf->stream($filename, array('Attachment' => 1));
                exit;
                
            } catch (Exception $e) {
                // If dompdf fails, show error
                die('Error generating PDF: ' . $e->getMessage() . '<br><br>Please ensure dompdf is installed: composer require dompdf/dompdf');
            }
        }
    }
}
