<?php
class SiswaController extends Controller {
    private $examModel;
    private $userModel;

    public function __construct() {
        AuthMiddleware::check(['siswa']);
        $this->examModel = $this->model('Exam');
        $this->userModel = $this->model('User');
    }

    public function profile() {
        $user = $this->userModel->getUserById($_SESSION['user_id']);
        $data = [
            'title' => 'Profil Saya',
            'active_menu' => 'profile',
            'user' => $user
        ];
        $this->view('layouts/header', $data);
        $this->view('profile', $data);
        $this->view('layouts/footer');
    }

    public function update_profile() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            if($_POST['id'] != $_SESSION['user_id']) die("Unauthorized");

            $password = trim($_POST['password']);
            $password_confirm = trim($_POST['password_confirm']);
            
            if(!empty($password) && $password !== $password_confirm) {
                set_flash('Konfirmasi password tidak cocok', 'danger');
                redirect('siswa/profile');
                return;
            }

            $data = [
                'id' => $_SESSION['user_id'],
                'name' => trim($_POST['name']),
                'username' => trim($_POST['username']),
                'role' => $_POST['role'], 
                'class_id' => $_POST['class_id'],
                'password' => $password
            ];

            if ($this->userModel->updateUser($data)) {
                 $_SESSION['name'] = $data['name'];
                 set_flash('Profil berhasil diperbarui', 'success');
            } else {
                 set_flash('Gagal memperbarui profil', 'danger');
            }
        }
        redirect('siswa/profile');
    }

    public function dashboard() {
        // Updated to only show exams for student's class
        // Assuming student class_id needs to be fetched or stored in session
        // Let's refetch user to be sure or assume session has it? 
        // For safety, let's update User Model to get user details including class_id, or trust session if we update login?
        // Since we didn't update login to store class_id in session yet, let's fetch user profile here.
        
        $userModel = $this->model('User');
        // We need a method in User model to get user by ID with class info, but for now let's query raw or use helper
        // Simplified: Fetch Exams by Class ID. 
        // We need to know student's class_id. 
        // Let's add getStudentClassId to ExamModel or User model.
        // Or query `users` table directly here? No, bad practice.
        
        // Quick fix: Get User Profile first
        // But wait, SiswaController doesn't load User model by default.
        // Let's rely on ExamModel to filter.
        
        $exams = $this->examModel->getPublishedExamsByClass($_SESSION['user_id']); // We will implement this new method in Exam model to lookup student class then exams
        // Check if student has already taken exams
        foreach ($exams as $exam) {
            $session = $this->examModel->checkExamSession($_SESSION['user_id'], $exam->id);
            if ($session) {
                $exam->session = $session;
            } else {
                $exam->session = null;
            }
        }

        $data = [
            'title' => 'Siswa Dashboard',
            'active_menu' => 'dashboard',
            'exams' => $exams
        ];
        $this->view('layouts/header', $data);
        $this->view('siswa/dashboard', $data);
        $this->view('layouts/footer');
    }

    public function join_exam($exam_id) {
        $exam = $this->examModel->getExamById($exam_id);
        if (!$exam) {
            set_flash('Ujian tidak ditemukan', 'danger');
            redirect('siswa/dashboard');
            return;
        }

        // Check if session already exists
        $session = $this->examModel->checkExamSession($_SESSION['user_id'], $exam_id);
        
        // If session exists, just resume (no token needed if already started)
        if ($session) {
             $session_id = $session->id;
             
             // STRICT: Block banned users
             if ($session->status == 'banned') {
                 set_flash('Anda telah diblokir dari ujian ini karena pelanggaran integritas.', 'danger');
                 redirect('siswa/dashboard');
                 return;
             }
             
             if ($session->status == 'completed') {
                 set_flash('Anda sudah menyelesaikan ujian ini.', 'info');
                 redirect('siswa/dashboard'); // Or show result
                 return;
             }
             redirect('siswa/take_exam/' . $session_id);
             return;
        }

        // If no session, require Token
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $input_token = trim($_POST['token']);
            if ($input_token === $exam->token) {
                // Token Correct -> Start Session
                $session_id = $this->examModel->startExamSession($_SESSION['user_id'], $exam_id);
                if ($session_id) {
                    redirect('siswa/take_exam/' . $session_id);
                } else {
                    set_flash('Gagal memulai sesi ujian', 'danger');
                    redirect('siswa/dashboard');
                }
            } else {
                set_flash('Token salah!', 'danger');
                // Reload view with error (fall through to view loading below)
            }
        }

        // Show Token Form
        $data = [
            'title' => 'Masukkan Token - ' . $exam->title,
            'active_menu' => 'dashboard',
            'exam' => $exam
        ];
        $this->view('layouts/header', $data);
        $this->view('siswa/exam_token', $data);
        $this->view('layouts/footer');
    }

    public function take_exam($session_id) {
        $session = $this->examModel->getExamSession($session_id);
        if (!$session || $session->student_id != $_SESSION['user_id']) {
            redirect('siswa/dashboard');
        }

        // STRICT: Block Banned Status
        if ($session->status == 'banned') {
            set_flash('Akses ditolak. Anda telah diblokir dari ujian ini.', 'danger');
            redirect('siswa/dashboard');
            return;
        }
        
        // ✅ Prevent re-entry if already completed
        if ($session->status == 'completed') {
            set_flash('Ujian sudah selesai. Anda tidak dapat mengakses kembali.', 'info');
            redirect('siswa/dashboard');
        }
        
        $exam = $this->examModel->getExamById($session->exam_id);
        $questions = $this->examModel->getQuestionsByExamId($session->exam_id);
        
        // Requirement: "Acak urutan soal"
        shuffle($questions);

        // Requirement: "Acak opsi jawaban" (Only for Pilihan Ganda)
        foreach ($questions as $q) {
            $type = isset($q->type) ? $q->type : (isset($q->jenis) ? $q->jenis : 'pilihan_ganda');
            
            if ($type == 'pilihan_ganda') {
                $q->options = $this->examModel->getOptionsByQuestionId($q->id); 
                shuffle($q->options);
            } elseif ($type == 'benar_salah') {
                $q->options = [
                    (object)['id' => 'true', 'teks_pilihan' => 'Benar'],
                    (object)['id' => 'false', 'teks_pilihan' => 'Salah']
                ];
            } elseif ($type == 'pilihan_ganda_kompleks' || $type == 'menjodohkan') {
                // Load options untuk tipe soal ini (no shuffle untuk konsistensi)
                $q->options = $this->examModel->getOptionsByQuestionId($q->id);
            }
        }

        $saved_answers = $this->examModel->getStudentAnswers($session_id);

        // Map saved answers for easy lookup
        $answers_map = [];
        foreach($saved_answers as $ans) {
            $answers_map[$ans->question_id] = $ans->selected_option;
        }

        // Calculate Remaining Seconds for Timer
        $current_time = time();
        $end_time = strtotime($session->end_time);
        $remainingSeconds = $end_time - $current_time;
        if($remainingSeconds < 0) $remainingSeconds = 0;

        $isBanned = ($session->status === 'banned');

        $data = [
            'title' => 'Taking Exam - ' . $exam->title,
            'exam' => $exam,
            'questions' => $questions,
            'session' => $session,
            'answers_map' => $answers_map,
            'remainingSeconds' => $remainingSeconds,
            'isBanned' => $isBanned
        ];

        // Custom view without standard header/footer to minimize distractions/navigation
        $this->view('siswa/take_exam', $data);
    }

    // AJAX Endpoint
    public function save_answer() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $data = json_decode(file_get_contents("php://input"));
            if($data) {
                $this->examModel->saveAnswer($data->session_id, $data->question_id, $data->answer);
                echo json_encode(['status' => 'success']);
            }
        }
    }
    
    // AJAX Endpoint for Ban
    public function ban_user() {
         if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $data = json_decode(file_get_contents("php://input"));
            if($data) {
                // Capture violation details
                $reason = isset($data->reason) ? $data->reason : 'Suspicious activity';
                $ip_address = $_SERVER['REMOTE_ADDR'];
                $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';
                
                // Ban student with full tracking
                $this->examModel->banStudent($data->session_id, $reason, $ip_address, $user_agent);
                
                // Return immediate ban confirmation
                echo json_encode([
                    'status' => 'banned',
                    'reason' => $reason,
                    'redirect' => BASE_URL . 'siswa/dashboard'
                ]);
            }
        }
    }

    // AJAX Endpoint for Activity Log
    public function log_activity() {
        if ($_SERVER['REQUEST_METHOD'] == 'POST') {
            $data = json_decode(file_get_contents("php://input"));
            if($data) {
                 // Use User Model to log
                 $userModel = $this->model('User');
                 $userModel->logActivity($_SESSION['user_id'], $data->activity, $data->details);
                 echo json_encode(['status' => 'logged']);
            }
        }
    }

    public function finish_exam($session_id) {
        $session = $this->examModel->getExamSession($session_id);
        if (!$session || $session->student_id != $_SESSION['user_id']) {
            redirect('siswa/dashboard');
        }
        
        // ✅ Allow banned users to finish exam - status will be updated to 'completed'
        // Log if user was banned before finishing
        if ($session->status == 'banned') {
            error_log("Student {$_SESSION['user_id']} finishing exam {$session_id} despite being banned. Will update status to completed.");
        }
        
        // ✅ Prevent re-entry only if already completed (not if banned)
        if ($session->status == 'completed') {
            set_flash('Ujian sudah selesai sebelumnya.', 'info');
            redirect('siswa/dashboard');
            return;
        }

        // Calculate Score (Auto-grade)
        $questions = $this->examModel->getQuestionsByExamId($session->exam_id);
        $answers = $this->examModel->getStudentAnswers($session_id);
        
        $ans_map = [];
        foreach($answers as $a) $ans_map[$a->question_id] = $a;

        $total_score_earned = 0;
        $total_max_score = 0;
        $has_manual_grading = false;

        foreach($questions as $q) {
            $q_max = 10; // Assume 10 points per question
            $total_max_score += $q_max;
            $q_earned = 0;

            if (isset($ans_map[$q->id])) {
                $user_ans_obj = $ans_map[$q->id];
                $user_val = $user_ans_obj->selected_option;
                $answer_id = $user_ans_obj->id;

                $type = isset($q->type) ? $q->type : (isset($q->jenis) ? $q->jenis : 'pilihan_ganda');

                if ($type == 'essay' || $type == 'uraian') {
                    $has_manual_grading = true;
                    $q_earned = 0; // Manual grading required
                } 
                elseif ($type == 'pilihan_ganda' || $type == 'benar_salah') {
                    // Check against is_correct in options (preferred) or answer_key (legacy)
                    $opts = $this->examModel->getOptionsByQuestionId($q->id);
                    $found_correct = false;
                    
                    // First check advanced options table
                    if (!empty($opts)) {
                        foreach($opts as $o) {
                            // Loose comparison for ID (string '15' vs int 15) vs String 'A'
                            if (trim($o->id) == trim($user_val) && !empty($o->is_correct)) {
                                $found_correct = true; break;
                            }
                        }
                    } 
                    // Fallback legacy (text based match)
                    if (!$found_correct && strcasecmp(trim($q->answer_key), trim($user_val)) == 0) {
                        $found_correct = true;
                    }
                    
                    if ($found_correct) $q_earned = $q_max;
                }
                elseif ($type == 'pilihan_ganda_kompleks') {
                    $user_ids = json_decode($user_val, true);
                    if (is_array($user_ids)) {
                        $opts = $this->examModel->getOptionsByQuestionId($q->id);
                        $correct_ids = [];
                        foreach($opts as $o) if($o->is_correct) $correct_ids[] = $o->id;

                        if (count($correct_ids) > 0) {
                             $matches = array_intersect($user_ids, $correct_ids);
                             $wrongs = array_diff($user_ids, $correct_ids);
                             $net = count($matches) - count($wrongs);
                             if ($net < 0) $net = 0;
                             $q_earned = ($net / count($correct_ids)) * $q_max;
                        }
                    }
                }
                elseif ($type == 'menjodohkan') {
                    $user_pairs = json_decode($user_val, true); // {opt_id: "pair_text"}
                    if (is_array($user_pairs)) {
                        $opts = $this->examModel->getOptionsByQuestionId($q->id);
                        $correct_count = 0;
                        $total_pairs = 0;
                        foreach($opts as $o) {
                            if (!empty($o->pair_text)) {
                                $total_pairs++;
                                if (isset($user_pairs[$o->id]) && $user_pairs[$o->id] === $o->pair_text) {
                                    $correct_count++;
                                }
                            }
                        }
                        if ($total_pairs > 0) {
                            $q_earned = ($correct_count / $total_pairs) * $q_max;
                        }
                    }
                }
                elseif ($type == 'isian_singkat' || $type == 'isian') {
                     // Get correct text. Usually in question_options (text) with is_correct=1 OR answer_key
                     // For 'isian_singkat', often stored in `answer_key` column in legacy or single correct option text.
                     // Let's check `answer_key` column first
                     $key = $q->answer_key;
                     if (strcasecmp(trim($user_val), trim($key)) == 0) {
                         $q_earned = $q_max;
                     }
                }

                $total_score_earned += $q_earned;
                $this->examModel->updateAnswerScore($answer_id, $q_earned);
            }
        }

        // Final Grade (0-100)
        $final_grade = ($total_max_score > 0) ? ($total_score_earned / $total_max_score) * 100 : 0;
        
        // If essay exists, the grade is tentative.
        $this->examModel->finishExamSession($session_id, $final_grade);

        $exam = $this->examModel->getExamById($session->exam_id);

        $data = [
            'title' => 'Ujian Selesai',
            'active_menu' => 'dashboard',
            'score' => $final_grade,
            'exam' => $exam,
            'has_manual_grading' => $has_manual_grading
        ];
        
        $this->view('layouts/header', $data);
        $this->view('siswa/exam_finish', $data);
        $this->view('layouts/footer');
    }
}
