<?php
class ServerMiddleware {
    private $db;
    private $userModel;
    private $schoolSettings;

    public function __construct() {
        $this->db = new Database();
        require_once '../app/models/User.php';
        require_once '../app/models/SchoolSettings.php';
        $this->userModel = new User();
        $this->schoolSettings = new SchoolSettings();
    }

    public static function checkMatrix() {
        $instance = new self();
        $instance->runChecks();
    }

    public function runChecks() {
        // 1. Update Last Active if User is Logged In
        if (isset($_SESSION['user_id'])) {
            $this->userModel->updateLastActive($_SESSION['user_id']);
            
            // 4. CHECK IF USER IS SUSPENDED
            $user = $this->userModel->getUserById($_SESSION['user_id']);
            if ($user && $user->status_akun === 'suspended') {
                if ($user->suspended_until && strtotime($user->suspended_until) > time()) {
                    // Check if request is not logout (to prevent redirect loop)
                    $url = isset($_GET['url']) ? $_GET['url'] : '';
                    if (strpos($url, 'auth/logout') === false) {
                        $this->showErrorPage("Akun Disuspend", "Akun Anda sedang dibekukan sementara hingga " . date('d M Y H:i', strtotime($user->suspended_until)) . ". Silakan hubungi admin.");
                    }
                } else {
                    // Auto unsuspend if time passed
                    $this->userModel->unsuspendUser($user->id);
                }
            }
        }

        // Get Settings
        $maintenance_mode = $this->schoolSettings->getSetting('maintenance_mode');
        $server_message = $this->schoolSettings->getSetting('server_message');
        
        // 2. Check Maintenance Mode
        if ($maintenance_mode === '1') {
            // Allow Admin to bypass
            if (isset($_SESSION['role']) && $_SESSION['role'] === 'admin') {
                return;
            }
            // Allow Login Page (so admins can login)
            $url = isset($_GET['url']) ? $_GET['url'] : '';
            if (strpos($url, 'auth/login') !== false || strpos($url, 'auth/process_login') !== false) {
                 return;
            }

            // Otherwise, show maintenance
            $this->showMaintenancePage($server_message);
        }

        // 3. Check Traffic Limit (Only for NEW logins or sessions?)
        // Better to check on login, but checking every request is safer for hard limits.
        // However, checking DB count on every request is heavy.
        // Optimization: Only check for non-admins and maybe cache result? 
        // For simple limited PHP app, query is fine.
        
        $max_users = $this->schoolSettings->getSetting('max_concurrent_users');
        if ($max_users && is_numeric($max_users) && $max_users > 0) {
             // Exclude admin from limit
             if (isset($_SESSION['role']) && $_SESSION['role'] === 'admin') return;

             // Don't block if user is already inside (active in last 5 mins)
             // But counting query already filters by activity.
             
             // Strategy: If user is trying to LOGIN (auth/process_login) AND count >= max, Block.
             // If user is already logged in, let them be? Or kick? usually let be.
             // But 'Server Limit' usually implies 'Concurrent Users'.
             
             $url = isset($_GET['url']) ? $_GET['url'] : '';
             // Only block NEW sessions or Login attempts if full
             if (strpos($url, 'auth/process_login') !== false) {
                  $active_count = $this->userModel->countActiveUsers(5);
                  if ($active_count >= $max_users) {
                       $this->showErrorPage("Server Penuh", "Maaf, jumlah pengguna saat ini mencapai batas maksimal ($max_users). Silakan coba lagi nanti.");
                  }
             }
        }
    }

    private function showMaintenancePage($message) {
        http_response_code(503);
        require_once '../app/views/errors/maintenance.php';
        exit;
    }

    private function showErrorPage($title, $message) {
        $data = ['title' => $title, 'message' => $message];
        require_once '../app/views/errors/generic.php';
        exit;
    }
}
