<?php
class Exam {
    private $db;

    public function __construct() {
        $this->db = new Database;
    }

    // Get all exams by specific guru
    // Get Guru Profile by User ID (for simplified schema)
    public function getGuruProfileByUserId($user_id) {
        // Simplified schema: query users table directly
        $this->db->query("SELECT * FROM users WHERE id = :uid AND role = 'guru'");
        $this->db->bind(':uid', $user_id);
        $user = $this->db->single();
        
        if($user) {
            // Return user data with id and name fields
            return (object)[
                'id' => $user->id, 
                'nama_lengkap' => $user->name,
                'username' => $user->username
            ]; 
        }
        return false;
    }

    public function getAllQuestionsByGuru($guru_id) {
        // Feature: Bank Soal - List questions owned by Guru
        $this->db->query("
            SELECT q.*, q.question_text as pertanyaan, 
            e.title as exam_title, e.id as ujian_id,
            s.name as nama_mapel,
            c.name as nama_kelas
            FROM questions q
            JOIN exams e ON q.exam_id = e.id
            LEFT JOIN subjects s ON e.subject_id = s.id
            LEFT JOIN classes c ON e.class_id = c.id
            WHERE e.guru_id = :gid
            ORDER BY e.created_at DESC
        ");
        
        $this->db->bind(':gid', $guru_id);
        return $this->db->resultSet();
    }

    public function getExamsByGuru($guru_id) {
        $this->db->query("SELECT * FROM exams WHERE guru_id = :guru_id ORDER BY created_at DESC");
        $this->db->bind(':guru_id', $guru_id);
        return $this->db->resultSet();
    }

    // Add new exam
    public function addExam($data) {
        $this->db->query("INSERT INTO exams (guru_id, title, description, duration_minutes, start_datetime, end_datetime, token, status, class_id, subject_id) VALUES (:guru_id, :title, :description, :duration_minutes, :start_datetime, :end_datetime, :token, :status, :class_id, :subject_id)");
        $this->db->bind(':guru_id', $data['guru_id']);
        $this->db->bind(':title', $data['title']);
        $this->db->bind(':description', $data['description']);
        $this->db->bind(':duration_minutes', $data['duration_minutes']);
        $this->db->bind(':start_datetime', $data['start_datetime']);
        $this->db->bind(':end_datetime', $data['end_datetime']);
        $this->db->bind(':token', $data['token']);
        $this->db->bind(':status', $data['status']);
        $this->db->bind(':class_id', $data['class_id']);
        $this->db->bind(':subject_id', $data['subject_id']);

        if ($this->db->execute()) {
            return true;
        } else {
            return false;
        }
    }

    // Update Exam
    public function updateExam($data) {
        $this->db->query("UPDATE exams SET 
            title = :title, 
            description = :description, 
            duration_minutes = :duration_minutes, 
            start_datetime = :start_datetime, 
            end_datetime = :end_datetime, 
            token = :token, 
            class_id = :class_id, 
            subject_id = :subject_id 
            WHERE id = :id");
            
        $this->db->bind(':id', $data['id']);
        $this->db->bind(':title', $data['title']);
        $this->db->bind(':description', $data['description']);
        $this->db->bind(':duration_minutes', $data['duration_minutes']);
        $this->db->bind(':start_datetime', $data['start_datetime']);
        $this->db->bind(':end_datetime', $data['end_datetime']);
        $this->db->bind(':token', $data['token']);
        $this->db->bind(':class_id', $data['class_id']);
        $this->db->bind(':subject_id', $data['subject_id']);

        if ($this->db->execute()) {
            return true;
        } else {
            return false;
        }
    }

    // Get single exam
    public function getExamById($id) {
        $this->db->query("SELECT * FROM exams WHERE id = :id");
        $this->db->bind(':id', $id);
        return $this->db->single();
    }
    
    // Delete Exam
    public function deleteExam($id) {
        $this->db->query("DELETE FROM exams WHERE id = :id");
        $this->db->bind(':id', $id);
        if ($this->db->execute()) {
            return true;
        } else {
            return false;
        }
    }

    // Update Exam Status
    public function updateStatus($id, $status) {
        $this->db->query("UPDATE exams SET status = :status WHERE id = :id");
        $this->db->bind(':status', $status);
        $this->db->bind(':id', $id);
        return $this->db->execute();
    }

    // --- Questions ---

    public function getQuestionsByExamId($exam_id) {
        $this->db->query("SELECT * FROM questions WHERE exam_id = :exam_id");
        $this->db->bind(':exam_id', $exam_id);
        return $this->db->resultSet();
    }

    public function addQuestion($data) {
        $this->db->query("INSERT INTO questions (exam_id, question_text, question_image, type, option_a, option_a_image, option_b, option_b_image, option_c, option_c_image, option_d, option_d_image, option_e, option_e_image, answer_key) VALUES (:exam_id, :question_text, :question_image, :type, :option_a, :option_a_image, :option_b, :option_b_image, :option_c, :option_c_image, :option_d, :option_d_image, :option_e, :option_e_image, :answer_key)");
        $this->db->bind(':exam_id', $data['exam_id']);
        $this->db->bind(':question_text', $data['question_text']);
        $this->db->bind(':question_image', isset($data['question_image']) ? $data['question_image'] : null);
        $this->db->bind(':type', isset($data['type']) ? $data['type'] : 'pilihan_ganda');
        
        // Bind text options (legacy)
        $this->db->bind(':option_a', isset($data['option_a']) ? $data['option_a'] : null);
        $this->db->bind(':option_b', isset($data['option_b']) ? $data['option_b'] : null);
        $this->db->bind(':option_c', isset($data['option_c']) ? $data['option_c'] : null);
        $this->db->bind(':option_d', isset($data['option_d']) ? $data['option_d'] : null);
        $this->db->bind(':option_e', isset($data['option_e']) ? $data['option_e'] : null);
        
        // Bind option images (NEW)
        $this->db->bind(':option_a_image', isset($data['option_a_image']) ? $data['option_a_image'] : null);
        $this->db->bind(':option_b_image', isset($data['option_b_image']) ? $data['option_b_image'] : null);
        $this->db->bind(':option_c_image', isset($data['option_c_image']) ? $data['option_c_image'] : null);
        $this->db->bind(':option_d_image', isset($data['option_d_image']) ? $data['option_d_image'] : null);
        $this->db->bind(':option_e_image', isset($data['option_e_image']) ? $data['option_e_image'] : null);
        
        // answer_key cannot be null, use '-' for types that don't satisfy it (like essay)
        $this->db->bind(':answer_key', !empty($data['answer_key']) ? $data['answer_key'] : '-'); 

        if ($this->db->execute()) {
            $question_id = $this->db->lastInsertId();

            // Handle Advanced Options (if provided as array)
            if (isset($data['advanced_options']) && is_array($data['advanced_options'])) {
                foreach ($data['advanced_options'] as $opt) {
                    $this->db->query("INSERT INTO question_options (question_id, option_text, option_image, is_correct, pair_text, pair_image) VALUES (:qid, :otext, :oimg, :is_correct, :ptext, :pimg)");
                    $this->db->bind(':qid', $question_id);
                    $this->db->bind(':otext', $opt['text']);
                    $this->db->bind(':oimg', isset($opt['image']) ? $opt['image'] : null);
                    $this->db->bind(':is_correct', isset($opt['is_correct']) ? 1 : 0);
                    $this->db->bind(':ptext', isset($opt['pair_text']) ? $opt['pair_text'] : null);
                    $this->db->bind(':pimg', isset($opt['pair_image']) ? $opt['pair_image'] : null);
                    $this->db->execute();
                }
            }
            return true;
        } else {
            return false;
        }
    }

    public function getOptionsByQuestionId($question_id) {
        // 1. Try fetching from new table
        $this->db->query("SELECT * FROM question_options WHERE question_id = :qid");
        $this->db->bind(':qid', $question_id);
        $advanced_opts = $this->db->resultSet();

        if (count($advanced_opts) > 0) {
            $options = [];
            foreach ($advanced_opts as $opt) {
                $options[] = (object) [
                    'id' => $opt->id, // Use DB ID as value for advanced types
                    'teks_pilihan' => $opt->option_text,
                    'image' => $opt->option_image,
                    'is_correct' => $opt->is_correct,
                    'pair_text' => $opt->pair_text,
                    'pair_image' => $opt->pair_image
                ];
            }
            return $options;
        }

        // 2. Fallback to Legacy Flat Columns (with image support NOW)
        $this->db->query("SELECT option_a, option_a_image, option_b, option_b_image, option_c, option_c_image, option_d, option_d_image, option_e, option_e_image, answer_key FROM questions WHERE id = :qid");
        $this->db->bind(':qid', $question_id);
        $row = $this->db->single();
        
        $options = [];
        if($row) {
            $map = ['a'=>'A', 'b'=>'B', 'c'=>'C', 'd'=>'D', 'e'=>'E'];
            foreach($map as $col => $label) {
                $key = "option_" . $col;
                $img_key = "option_" . $col . "_image";
                if(!empty($row->$key)) {
                    $options[] = (object) [
                        'id' => $label, 
                        'teks_pilihan' => $row->$key,
                        'image' => isset($row->$img_key) ? $row->$img_key : null, // NOW includes images
                        'pair_text' => null,
                        'is_correct' => (strcasecmp($row->answer_key, $label) == 0) ? 1 : 0
                    ];
                }
            }
        }
        return $options;
    }

    public function getExamResults($exam_id) {
        // Simplified schema: use school_class directly from users table
        $this->db->query("
            SELECT ep.*, u.name as student_name, u.username, 
            COALESCE(u.school_class, '-') as class_name
            FROM exam_participants ep
            JOIN users u ON ep.student_id = u.id
            WHERE ep.exam_id = :eid
            ORDER BY class_name ASC, student_name ASC
        ");
        $this->db->bind(':eid', $exam_id);
        return $this->db->resultSet();
    }

    public function updateAnswerScore($id, $score) {
        $this->db->query("UPDATE student_answers SET score = :score WHERE id = :id");
        $this->db->bind(':score', $score);
        $this->db->bind(':id', $id);
        return $this->db->execute();
    }

    public function updateShowResults($exam_id, $status) {
        $this->db->query("UPDATE exams SET show_results = :status WHERE id = :id");
        $this->db->bind(':status', $status);
        $this->db->bind(':id', $exam_id);
        return $this->db->execute();
    }

    public function deleteQuestion($id) {
        $this->db->query("DELETE FROM questions WHERE id = :id");
        $this->db->bind(':id', $id);
        if ($this->db->execute()) {
            return true;
        } else {
            return false;
        }
    }

    // --- Student Exam Session ---

    public function getPublishedExams() {
        $this->db->query("SELECT users.name as guru_name, exams.* FROM exams JOIN users ON exams.guru_id = users.id WHERE exams.status = 'published' ORDER BY exams.created_at DESC");
        return $this->db->resultSet();
    }

    public function getPublishedExamsByClass($student_id) {
        // 1. Get Student Class ID
        $this->db->query("SELECT class_id FROM users WHERE id = :id");
        $this->db->bind(':id', $student_id);
        $student = $this->db->single();

        if (!$student || !$student->class_id) {
            return []; 
        }

        // 2. Get Exams for this class
        $this->db->query("SELECT users.name as guru_name, exams.* 
                          FROM exams 
                          JOIN users ON exams.guru_id = users.id 
                          WHERE exams.status = 'published' AND exams.class_id = :class_id 
                          ORDER BY exams.created_at DESC");
        $this->db->bind(':class_id', $student->class_id);
        return $this->db->resultSet();
    }

    public function checkExamSession($student_id, $exam_id) {
        $this->db->query("SELECT * FROM exam_participants WHERE student_id = :student_id AND exam_id = :exam_id");
        $this->db->bind(':student_id', $student_id);
        $this->db->bind(':exam_id', $exam_id);
        return $this->db->single();
    }

    public function startExamSession($student_id, $exam_id) {
        $exam = $this->getExamById($exam_id);
        $duration = $exam->duration_minutes;
        $start_time = date('Y-m-d H:i:s');
        $end_time = date('Y-m-d H:i:s', strtotime("+$duration minutes"));

        $this->db->query("INSERT INTO exam_participants (exam_id, student_id, start_time, end_time, status) VALUES (:exam_id, :student_id, :start_time, :end_time, 'ongoing')");
        $this->db->bind(':exam_id', $exam_id);
        $this->db->bind(':student_id', $student_id);
        $this->db->bind(':start_time', $start_time);
        $this->db->bind(':end_time', $end_time);
        
        if($this->db->execute()) {
             return $this->db->lastInsertId();
        } else {
             return false;
        }
    }
    
    public function getExamSession($session_id) {
         $this->db->query("SELECT * FROM exam_participants WHERE id = :id");
         $this->db->bind(':id', $session_id);
         return $this->db->single();
    }

    public function saveAnswer($participant_id, $question_id, $selected_option) {
        $this->db->query("SELECT * FROM student_answers WHERE participant_id = :participant_id AND question_id = :question_id");
        $this->db->bind(':participant_id', $participant_id);
        $this->db->bind(':question_id', $question_id);
        $existing = $this->db->single();

        if ($existing) {
            $this->db->query("UPDATE student_answers SET selected_option = :selected_option WHERE id = :id");
            $this->db->bind(':selected_option', $selected_option);
            $this->db->bind(':id', $existing->id);
        } else {
            $this->db->query("INSERT INTO student_answers (participant_id, question_id, selected_option) VALUES (:participant_id, :question_id, :selected_option)");
            $this->db->bind(':participant_id', $participant_id);
            $this->db->bind(':question_id', $question_id);
            $this->db->bind(':selected_option', $selected_option);
        }
        return $this->db->execute();
    }

    public function getStudentAnswers($participant_id) {
        $this->db->query("SELECT * FROM student_answers WHERE participant_id = :participant_id");
        $this->db->bind(':participant_id', $participant_id);
        return $this->db->resultSet();
    }
    
    public function banStudent($participant_id, $reason, $ip_address = null, $user_agent = null) {
        // Update participant status to banned
        $this->db->query("UPDATE exam_participants SET status = 'banned', ban_reason = :reason WHERE id = :id");
        $this->db->bind(':reason', $reason);
        $this->db->bind(':id', $participant_id);
        $result = $this->db->execute();
        
        // Log the violation
        if ($result) {
            $this->logViolation($participant_id, $reason, $reason, $ip_address, $user_agent);
            
            // Update violation count and first violation time
            $this->db->query("UPDATE exam_participants SET 
                violation_count = violation_count + 1,
                first_violation_time = COALESCE(first_violation_time, NOW())
                WHERE id = :id");
            $this->db->bind(':id', $participant_id);
            $this->db->execute();
        }
        
        return $result;
    }
    
    public function logViolation($participant_id, $violation_type, $violation_details, $ip_address = null, $user_agent = null) {
        $this->db->query("INSERT INTO violation_logs (participant_id, violation_type, violation_details, ip_address, user_agent) 
                         VALUES (:participant_id, :violation_type, :violation_details, :ip_address, :user_agent)");
        $this->db->bind(':participant_id', $participant_id);
        $this->db->bind(':violation_type', $violation_type);
        $this->db->bind(':violation_details', $violation_details);
        $this->db->bind(':ip_address', $ip_address);
        $this->db->bind(':user_agent', $user_agent);
        return $this->db->execute();
    }
    
    public function getViolationHistory($participant_id) {
        $this->db->query("SELECT * FROM violation_logs WHERE participant_id = :participant_id ORDER BY timestamp DESC");
        $this->db->bind(':participant_id', $participant_id);
        return $this->db->resultSet();
    }

    public function finishExamSession($session_id, $score) {
        // Force status update to 'completed' regardless of previous status (including 'banned')
        // This ensures banned students can still complete and submit their exams
        $this->db->query("UPDATE exam_participants SET status = 'completed', score = :score, end_time = NOW() WHERE id = :id");
        $this->db->bind(':score', $score);
        $this->db->bind(':id', $session_id);
        
        $result = $this->db->execute();
        
        // Log the status change for debugging
        if ($result) {
            error_log("Exam session {$session_id} finished successfully. Status updated to 'completed' with score: {$score}");
        } else {
            error_log("Failed to finish exam session {$session_id}");
        }
        
        return $result;
    }

    public function countExamsByStatus($status) {
        $this->db->query("SELECT COUNT(*) as count FROM exams WHERE status = :status");
        $this->db->bind(':status', $status);
        $row = $this->db->single();
        return $row ? $row->count : 0;
    }


    public function countCompletedSessions() {
        $this->db->query("SELECT COUNT(*) as count FROM exam_participants WHERE status = 'completed'");
        $row = $this->db->single();
        return $row ? $row->count : 0;
    }
    
    public function getAverageScoresByExam($limit = 10) {
        $this->db->query("
            SELECT 
                e.title as exam_title,
                e.created_at,
                AVG(ep.score) as avg_score,
                COUNT(ep.id) as participant_count
            FROM exams e
            LEFT JOIN exam_participants ep ON e.id = ep.exam_id AND ep.status = 'completed'
            WHERE e.status = 'published'
            GROUP BY e.id, e.title, e.created_at
            HAVING participant_count > 0
            ORDER BY e.created_at DESC
            LIMIT :limit
        ");
        $this->db->bind(':limit', $limit);
        return $this->db->resultSet();
    }
}
