<?php
class User {
    private $db;

    public function __construct() {
        $this->db = new Database;
    }

    public function login($username, $password) {
        $this->db->query("SELECT * FROM users WHERE username = :username");
        $this->db->bind(':username', $username);
        $user = $this->db->single();

        if ($user) {
            // Verify Password
            if (password_verify($password, $user->password)) {
                 // Check Status (if column exists)
                 if (isset($user->status_akun) && $user->status_akun != 'active') {
                     // Auto-Unsuspend Check
                     if ($user->status_akun == 'suspended' && !empty($user->suspended_until)) {
                         if (strtotime($user->suspended_until) <= time()) {
                             $this->unsuspendUser($user->id); // Reactivate
                             $user->status_akun = 'active';   // Allow login
                         } else {
                             return 'NOT_ACTIVE';
                         }
                     } else {
                         return 'NOT_ACTIVE';
                     }
                 }

                 // User data is already complete in the users table
                 // No need to query separate tables since we use simplified schema
                 return $user;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }
    
    // Check if username exists
    public function findUserByUsername($username) {
        $this->db->query("SELECT * FROM users WHERE username = :username");
        $this->db->bind(':username', $username);
        if ($this->db->rowCount() > 0) {
            return true;
        } else {
            return false;
        }
    }

    public function getAllUsers() {
        // Updated to join with classes and check for banned sessions AND activity stats
        // We prioritize showing 'ongoing' exam session details.
        // Timestamp check for online: 5 minutes.
        
        $this->db->query("SELECT u.*, c.name as class_name,
                          (SELECT COUNT(*) FROM exam_participants ep WHERE ep.student_id = u.id AND ep.status = 'banned') as banned_count,
                          IF(u.last_activity > DATE_SUB(NOW(), INTERVAL 5 MINUTE), 1, 0) as is_online,
                          
                          -- Active Session Details (Ongoing)
                          (SELECT ep.id FROM exam_participants ep WHERE ep.student_id = u.id AND ep.status = 'ongoing' ORDER BY ep.id DESC LIMIT 1) as active_session_id,
                          
                          -- Answered Count for Active Session
                          (SELECT COUNT(*) FROM student_answers sa 
                           WHERE sa.participant_id = (SELECT id FROM exam_participants ep2 WHERE ep2.student_id = u.id AND ep2.status = 'ongoing' ORDER BY ep2.id DESC LIMIT 1)
                          ) as answered_count,
                          
                          -- Total Questions for Active Session's Exam
                          (SELECT COUNT(*) FROM questions q 
                           WHERE q.exam_id = (SELECT exam_id FROM exam_participants ep3 WHERE ep3.student_id = u.id AND ep3.status = 'ongoing' ORDER BY ep3.id DESC LIMIT 1)
                          ) as total_questions

                          FROM users u 
                          LEFT JOIN classes c ON u.class_id = c.id 
                          ORDER BY u.created_at DESC");
        return $this->db->resultSet();
    }

    public function addUser($data) {
        // Hash password securely
        $hashed_password = password_hash($data['password'], PASSWORD_DEFAULT);
        
        $this->db->query("INSERT INTO users (name, username, password, role, class_id, nisn, tempat_lahir, tanggal_lahir, jenis_kelamin, alamat, telepon, foto) 
            VALUES (:name, :username, :password, :role, :class_id, :nisn, :tempat_lahir, :tanggal_lahir, :jenis_kelamin, :alamat, :telepon, :foto)");
        $this->db->bind(':name', $data['name']);
        $this->db->bind(':username', $data['username']);
        $this->db->bind(':password', $hashed_password);
        $this->db->bind(':role', $data['role']);
        $this->db->bind(':class_id', $data['class_id']);
        
        // Biodata fields
        $this->db->bind(':nisn', $data['nisn'] ?? null);
        $this->db->bind(':tempat_lahir', $data['tempat_lahir'] ?? null);
        $this->db->bind(':tanggal_lahir', $data['tanggal_lahir'] ?? null);
        $this->db->bind(':jenis_kelamin', $data['jenis_kelamin'] ?? null);
        $this->db->bind(':alamat', $data['alamat'] ?? null);
        $this->db->bind(':telepon', $data['telepon'] ?? null);
        $this->db->bind(':foto', $data['foto'] ?? null);

        // Check if username already exists (constraint violation)
        try {
            if ($this->db->execute()) {
                return true;
            } else {
                return false;
            }
        } catch (PDOException $e) {
            return false;
        }
    }

    public function deleteUser($id) {
        $this->db->query("DELETE FROM users WHERE id = :id");
        $this->db->bind(':id', $id);
        return $this->db->execute();
    }

    public function resetBannedSessions($user_id) {
         // Reset 'banned' status to 'ongoing' for specific user in exam_participants
         // Requirement: "reset admin tanpa kehilangan jawaban"
         
         // First check if there is a banned session
         $this->db->query("UPDATE exam_participants SET status = 'ongoing', ban_reason = NULL WHERE student_id = :student_id AND status = 'banned'");
         $this->db->bind(':student_id', $user_id);
         return $this->db->execute();
    }

    public function countByRole($role) {
        $this->db->query("SELECT COUNT(*) as count FROM users WHERE role = :role");
        $this->db->bind(':role', $role);
        $row = $this->db->single();
        return $row ? $row->count : 0;
    }

    public function logActivity($user_id, $activity, $keterangan = '') {
        $this->db->query("INSERT INTO log_aktivitas (user_id, aktivitas, keterangan, ip_address) VALUES (:uid, :act, :ket, :ip)");
        $this->db->bind(':uid', $user_id);
        $this->db->bind(':act', $activity);
        $this->db->bind(':ket', $keterangan);
        $this->db->bind(':ip', $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1');
        return $this->db->execute();
    }

    public function getLogs() {
        $this->db->query("SELECT l.*, u.username, u.role, u.name as user_name FROM log_aktivitas l JOIN users u ON l.user_id = u.id ORDER BY l.created_at DESC LIMIT 100");
        return $this->db->resultSet();
    }
    
    // Get students by class ID (for printing cards)
    public function getStudentsByClass($class_id) {
        $this->db->query("SELECT * FROM users WHERE role = 'siswa' AND class_id = :class_id ORDER BY name ASC");
        $this->db->bind(':class_id', $class_id);
        return $this->db->resultSet();
    }

    // Get Single User
    public function getUserById($id) {
        $this->db->query("SELECT * FROM users WHERE id = :id");
        $this->db->bind(':id', $id);
        return $this->db->single();
    }

    // Update User
    public function updateUser($data) {
        // Prepare base query
        $query = "UPDATE users SET 
            name = :name, 
            username = :username, 
            role = :role, 
            class_id = :class_id,
            nisn = :nisn,
            tempat_lahir = :tempat_lahir,
            tanggal_lahir = :tanggal_lahir,
            jenis_kelamin = :jenis_kelamin,
            alamat = :alamat,
            telepon = :telepon";

        // Add password to query if provided
        if (!empty($data['password'])) {
            $query .= ", password = :password";
        }
        
        // Add photo to query if provided
        if (!empty($data['foto'])) {
             $query .= ", foto = :foto";
        }

        $query .= " WHERE id = :id";

        $this->db->query($query);
        
        // Bind basic fields
        $this->db->bind(':id', $data['id']);
        $this->db->bind(':name', $data['name']);
        $this->db->bind(':username', $data['username']);
        $this->db->bind(':role', $data['role']);
        $this->db->bind(':class_id', $data['class_id']);
        
        // Bind biodata fields (use null if not set)
        $this->db->bind(':nisn', $data['nisn'] ?? null);
        $this->db->bind(':tempat_lahir', $data['tempat_lahir'] ?? null);
        $this->db->bind(':tanggal_lahir', $data['tanggal_lahir'] ?? null);
        $this->db->bind(':jenis_kelamin', $data['jenis_kelamin'] ?? null);
        $this->db->bind(':alamat', $data['alamat'] ?? null);
        $this->db->bind(':telepon', $data['telepon'] ?? null);

        if (!empty($data['password'])) {
            $hashed_password = password_hash($data['password'], PASSWORD_DEFAULT);
            $this->db->bind(':password', $hashed_password);
        }
        
        if (!empty($data['foto'])) {
            $this->db->bind(':foto', $data['foto']);
        }

    }

    public function updateLastActive($user_id) {
        $this->db->query("UPDATE users SET last_active = NOW() WHERE id = :id");
        $this->db->bind(':id', $user_id);
        return $this->db->execute();
    }

    public function countActiveUsers($minutes = 5) {
        $this->db->query("SELECT COUNT(*) as count FROM users WHERE last_active > DATE_SUB(NOW(), INTERVAL :minutes MINUTE)");
        $this->db->bind(':minutes', $minutes);
        $row = $this->db->single();
        return $row ? $row->count : 0;
    }
    public function getActiveUsersList($minutes = 5) {
        $this->db->query("SELECT u.id, u.username, u.name, u.role, u.last_active, u.status_akun, u.suspended_until, c.name as class_name 
                          FROM users u
                          LEFT JOIN classes c ON u.class_id = c.id
                          WHERE u.last_active > DATE_SUB(NOW(), INTERVAL :minutes MINUTE)
                          ORDER BY u.last_active DESC");
        $this->db->bind(':minutes', $minutes);
        return $this->db->resultSet();
    }

    public function suspendUser($user_id, $minutes) {
        $this->db->query("UPDATE users SET status_akun = 'suspended', suspended_until = DATE_ADD(NOW(), INTERVAL :minutes MINUTE) WHERE id = :id");
        $this->db->bind(':id', $user_id);
        $this->db->bind(':minutes', $minutes);
        return $this->db->execute();
    }
    
    public function unsuspendUser($user_id) {
        $this->db->query("UPDATE users SET status_akun = 'active', suspended_until = NULL WHERE id = :id");
        $this->db->bind(':id', $user_id);
        return $this->db->execute();
    }
}
