-- Database: dbujian
CREATE DATABASE IF NOT EXISTS `dbujian`;
USE `dbujian`;

-- 1. Users & Auth
CREATE TABLE IF NOT EXISTS `users` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `username` VARCHAR(50) NOT NULL UNIQUE,
    `password` VARCHAR(255) NOT NULL,
    `name` VARCHAR(100) NOT NULL,
    `role` ENUM('admin', 'guru', 'siswa') NOT NULL,
    `school_class` VARCHAR(50) DEFAULT NULL, -- Class for students
    `status_akun` ENUM('active', 'suspended', 'banned') DEFAULT 'active',
    `last_active` DATETIME DEFAULT NULL,
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- 2. Exams
CREATE TABLE IF NOT EXISTS `exams` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `guru_id` INT NOT NULL,
    `title` VARCHAR(200) NOT NULL,
    `description` TEXT,
    `duration_minutes` INT NOT NULL,
    `start_datetime` DATETIME NOT NULL,
    `end_datetime` DATETIME NOT NULL,
    `token` VARCHAR(20) DEFAULT NULL,
    `status` ENUM('draft', 'published', 'closed') DEFAULT 'draft',
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`guru_id`) REFERENCES `users`(`id`) ON DELETE CASCADE
);

-- 3. Questions
-- Supports Multiple Choice (options A-E) and Essay/TrueFalse (options ignored or used for T/F)
CREATE TABLE IF NOT EXISTS `questions` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `exam_id` INT NOT NULL,
    `question_text` TEXT NOT NULL,
    `jenis` ENUM('pilihan_ganda', 'essay', 'benar_salah') DEFAULT 'pilihan_ganda',
    `bobot` INT DEFAULT 1,
    `question_image` VARCHAR(255) DEFAULT NULL,
    -- Simple flat options for PG
    `option_a` TEXT,
    `option_b` TEXT,
    `option_c` TEXT,
    `option_d` TEXT,
    `option_e` TEXT,
    `answer_key` VARCHAR(255) NOT NULL, -- 'A', 'B', 'true', etc. or Essay keywords
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`exam_id`) REFERENCES `exams`(`id`) ON DELETE CASCADE
);

-- 4. Exam Participants (Sessions)
CREATE TABLE IF NOT EXISTS `exam_participants` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `exam_id` INT NOT NULL,
    `student_id` INT NOT NULL,
    `start_time` DATETIME DEFAULT NULL,
    `end_time` DATETIME DEFAULT NULL,
    `score` DECIMAL(5,2) DEFAULT NULL,
    `status` ENUM('ongoing', 'completed', 'banned') DEFAULT 'ongoing',
    `ban_reason` VARCHAR(255) DEFAULT NULL,
    FOREIGN KEY (`exam_id`) REFERENCES `exams`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`student_id`) REFERENCES `users`(`id`) ON DELETE CASCADE,
    UNIQUE KEY `unique_attempt` (`exam_id`, `student_id`)
);

-- 5. Student Answers
CREATE TABLE IF NOT EXISTS `student_answers` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `participant_id` INT NOT NULL,
    `question_id` INT NOT NULL,
    `selected_option` TEXT DEFAULT NULL, -- 'A', 'B', 'true', or Essay answer
    `is_correct` TINYINT(1) DEFAULT 0,
    FOREIGN KEY (`participant_id`) REFERENCES `exam_participants`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`question_id`) REFERENCES `questions`(`id`) ON DELETE CASCADE
);

-- 6. Activity Logs
CREATE TABLE IF NOT EXISTS `log_aktivitas` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT NOT NULL,
    `aktivitas` VARCHAR(255) NOT NULL,
    `keterangan` TEXT,
    `ip_address` VARCHAR(45),
    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE
);

-- Other tables like mata_pelajaran/kelas are optional if we stick to school_class string in users
-- But if code uses joins (e.g. getExamStats), they might be needed.
-- Adding simplistic versions just in case code references them indirectly (or we fixed code to relax).
-- Based on current 'User.php', 'Exam.php' we rely mostly on strings in 'users' or existing joins.
-- Code in Exam.php getExamResults does LEFT JOIN kelas.
CREATE TABLE IF NOT EXISTS `kelas` (
    `id` INT AUTO_INCREMENT PRIMARY KEY,
    `nama_kelas` VARCHAR(50) NOT NULL
);

-- Default Data
INSERT INTO `users` (`username`, `password`, `name`, `role`, `school_class`) VALUES 
('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Administrator', 'admin', NULL),
('guru', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Pak Budi', 'guru', NULL),
('siswa', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'Ani Siswa', 'siswa', 'XII IPA 1');
