// exam.js - Comprehensive Exam Engine

// ===== GLOBAL VARIABLES =====
let sessionId;
let currentQuestionIndex = 0;
let totalQuestions = 0;

// ===== GLOBAL NAVIGATION FUNCTIONS (Accessible from inline onclick) =====

window.jumpToQuestion = function (index) {
    if (index < 0 || index >= totalQuestions) return;

    // Hide current
    const currentCard = document.getElementById(`question-card-${currentQuestionIndex}`);
    const currentBtn = document.getElementById(`nav-btn-${currentQuestionIndex}`);
    if (currentCard) currentCard.style.display = 'none';
    if (currentBtn) currentBtn.classList.remove('btn-primary', 'text-white');

    // Restore style for previous question nav button
    updateNavStatus(currentQuestionIndex);

    // Show new
    currentQuestionIndex = index;
    const newCard = document.getElementById(`question-card-${currentQuestionIndex}`);
    if (newCard) newCard.style.display = 'block';

    // Update header number
    const headerNum = document.getElementById('current-question-num');
    if (headerNum) headerNum.textContent = currentQuestionIndex + 1;

    // Highlight active nav button
    const btn = document.getElementById(`nav-btn-${currentQuestionIndex}`);
    if (btn) {
        btn.classList.remove('btn-danger', 'btn-success', 'btn-warning', 'text-dark');
        btn.classList.add('btn-primary', 'text-white');
    }

    // Scroll top
    window.scrollTo(0, 0);
};

window.nextQuestion = function () {
    console.log('Next button clicked, current:', currentQuestionIndex);
    jumpToQuestion(currentQuestionIndex + 1);
};

window.prevQuestion = function () {
    console.log('Prev button clicked, current:', currentQuestionIndex);
    jumpToQuestion(currentQuestionIndex - 1);
};

window.updateNavStatus = function (index) {
    const btn = document.getElementById(`nav-btn-${index}`);
    if (!btn) return;
    if (index === currentQuestionIndex) return; // Don't style active button

    const isRagu = document.getElementById(`ragu_check_${index}`) && document.getElementById(`ragu_check_${index}`).checked;
    const hasAnswer = checkAnswerStatus(index);

    btn.className = 'btn w-100'; // Reset

    if (isRagu) {
        btn.classList.add('btn-warning', 'text-dark');
    } else if (hasAnswer) {
        btn.classList.add('btn-success', 'text-white');
    } else {
        // Changed from btn-outline-secondary to btn-danger (red for unanswered)
        btn.classList.add('btn-danger', 'text-white');
    }
};

window.toggleRagu = function (index) {
    console.log('Toggle Ragu clicked for question:', index);
    const checkbox = document.getElementById(`ragu_check_${index}`);
    if (!checkbox) return;

    checkbox.checked = !checkbox.checked;

    // Visual indicator on current card
    const card = document.getElementById(`question-card-${index}`);
    if (card) {
        if (checkbox.checked) {
            card.classList.add('border-warning');
            card.style.borderLeft = '5px solid #ffc107';
        } else {
            card.classList.remove('border-warning');
            card.style.borderLeft = 'none';
        }
    }
};

// ✅ Confirmation before finish
window.confirmFinish = function () {
    const agreeCheckbox = document.getElementById('agreeFinish');

    if (!agreeCheckbox.checked) {
        alert('Anda harus mencentang persetujuan terlebih dahulu!');
        return;
    }

    // ✅ CRITICAL FIX: Set isFinishing flag IMMEDIATELY to stop anti-cheat
    window.isFinishing = true;

    // Close modal
    const modalEl = document.getElementById('confirmFinishModal');
    const modal = bootstrap.Modal.getInstance(modalEl);
    if (modal) modal.hide();

    // Proceed to finish
    finishExam('Selesai mengerjakan');
};

window.finishExam = function (reason) {
    if (typeof BASE_URL === 'undefined') {
        console.error('BASE_URL is not defined!');
        alert('Error: BASE_URL tidak terdefinisi. Silakan refresh halaman.');
        return;
    }

    // ✅ Set flag to prevent anti-cheat triggers
    window.isFinishing = true;

    logActivity('Selesai Ujian', 'Reason: ' + reason);
    setTimeout(() => {
        alert(reason ? reason : "Menyimpan jawaban dan mengakhiri ujian...");
        window.location.href = BASE_URL + "siswa/finish_exam/" + sessionId;
    }, 500);
};

// ===== HELPER FUNCTIONS =====

function checkAnswerStatus(index) {
    const id = getQuestionId(index);
    if (!id) return false;

    // Radios / Checkboxes
    if (document.querySelectorAll(`input[name^="q_${id}"]:checked`).length > 0) return true;

    // Textarea / Text Input
    const texts = document.querySelectorAll(`textarea[name="q_${id}"], input[name="q_${id}"][type="text"]`);
    for (let t of texts) {
        if (t.value.trim().length > 0) return true;
    }

    // Selects (Matching)
    const selects = document.querySelectorAll(`select[name^="q_${id}"]`);
    for (let s of selects) {
        if (s.value !== "") return true;
    }

    return false;
}

function getQuestionId(index) {
    const card = document.getElementById(`question-card-${index}`);
    if (!card) return 0;

    // Find any input with name starting with q_
    const input = card.querySelector('[name^="q_"]');
    if (input) {
        const match = input.name.match(/^q_(\d+)/);
        if (match) return match[1];
    }
    return 0;
}

function logActivity(activity, details) {
    if (typeof BASE_URL === 'undefined') {
        console.error('BASE_URL not defined, cannot log activity');
        return;
    }

    fetch(BASE_URL + 'siswa/log_activity', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
            activity: activity,
            details: details + " [SessionID: " + sessionId + "]"
        })
    }).catch(err => console.error('Log activity failed:', err));
}

// ===== DOM READY INITIALIZATION =====

document.addEventListener('DOMContentLoaded', function () {
    console.log('Exam page loaded');

    // Check BASE_URL
    if (typeof BASE_URL === 'undefined') {
        console.error('CRITICAL: BASE_URL is not defined!');
        alert('Error: Konfigurasi BASE_URL tidak ditemukan. Halaman mungkin tidak berfungsi dengan baik.');
    } else {
        console.log('BASE_URL:', BASE_URL);
    }

    const sessionIdEl = document.getElementById('session_id');
    const timerDisplay = document.getElementById('timer-display');
    const examForm = document.getElementById('exam-form');

    if (!sessionIdEl) {
        console.error('session_id element not found!');
        return;
    }

    sessionId = sessionIdEl.value;
    const questions = document.querySelectorAll('.question-card');
    totalQuestions = questions.length;

    console.log('Total questions:', totalQuestions);

    // --- 1. Real-time Countdown Timer & Auto Submit ---
    if (timerDisplay) {
        const remainingSecondsEl = document.getElementById('remaining_seconds');
        let distance = parseInt(remainingSecondsEl ? remainingSecondsEl.value : 0) * 1000;

        const timerInterval = setInterval(function () {
            distance -= 1000;

            if (distance < 0) {
                clearInterval(timerInterval);
                timerDisplay.innerHTML = "WAKTU HABIS";
                timerDisplay.classList.add('text-danger', 'blink');

                // Set flag before finishing
                window.isFinishing = true;
                finishExam('Waktu Habis');
                return;
            }

            const hours = Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
            const minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
            const seconds = Math.floor((distance % (1000 * 60)) / 1000);

            timerDisplay.innerHTML = `<i class="fas fa-clock"></i> ${hours}:${minutes}:${seconds}`;

            if (distance < 300000) { // 5 mins
                timerDisplay.classList.add('text-warning');
            }
        }, 1000);
    }

    // --- 2. Auto-Save Logic (Instant + Periodic) ---
    if (examForm) {
        examForm.addEventListener('change', function (e) {
            const el = e.target;
            if (el.name && el.name.startsWith('q_')) {
                handleAnswerChange(el);
            }
        });
    }

    function handleAnswerChange(element) {
        const rawName = element.name;
        const match = rawName.match(/^q_(\d+)/);
        if (!match) return;

        const qId = match[1];
        let answerValue = null;

        if (element.type === 'checkbox') {
            const checkedBoxes = document.querySelectorAll(`input[name^="q_${qId}"]:checked`);
            const values = Array.from(checkedBoxes).map(cb => cb.value);
            answerValue = JSON.stringify(values);
        } else if (element.tagName === 'SELECT') {
            const selects = document.querySelectorAll(`select[name^="q_${qId}"]`);
            const pairs = {};
            selects.forEach(sel => {
                const pairMatch = sel.name.match(/\[(\d+)\]/);
                if (pairMatch) {
                    pairs[pairMatch[1]] = sel.value;
                }
            });
            answerValue = JSON.stringify(pairs);
        } else {
            answerValue = element.value;
        }

        saveSingleAnswer(qId, answerValue);
    }

    // Periodic: Save Essay Textareas
    const textareas = document.querySelectorAll('textarea, input[type="text"]');
    setInterval(function () {
        textareas.forEach(el => {
            if (el.name && el.name.startsWith('q_')) {
                const qId = el.name.match(/^q_(\d+)/)[1];
                saveSingleAnswer(qId, el.value);
            }
        });
        console.log("Auto-saving text...");
    }, 10000);

    function saveSingleAnswer(questionId, answer) {
        if (typeof BASE_URL === 'undefined') return;

        fetch(BASE_URL + 'siswa/save_answer', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                session_id: sessionId,
                question_id: questionId,
                answer: answer
            })
        }).then(res => res.json())
            .then(data => { /* Silent success */ })
            .catch(err => console.error('Save failed', err));
    }

    // --- 3. Strict Anti-Cheat with Runtime DevTools Detection ---
    history.pushState(null, null, location.href);
    window.onpopstate = function () {
        history.go(1);
        banUser("Menekan Tombol Back");
    };

    let isBanning = false;
    let devToolsOpen = false;

    // Enhanced Ban Function with Complete Screen Block
    function banUser(reason) {
        if (isBanning) return;
        isBanning = true;
        devToolsOpen = true; // Prevent further detection triggers

        if (typeof BASE_URL === 'undefined') {
            console.error('Cannot ban user, BASE_URL not defined');
            return;
        }

        console.log('⛔ VIOLATION DETECTED:', reason);

        // Log activity before banning
        logActivity('PELANGGARAN TERDETEKSI', reason);

        // Completely disable all interactions immediately
        document.body.style.pointerEvents = 'none';
        document.body.style.userSelect = 'none';
        document.body.style.overflow = 'hidden';

        // Create impenetrable full-screen blocking overlay
        const overlay = document.createElement('div');
        overlay.id = 'ban-overlay';
        overlay.style.cssText = `
            position: fixed !important;
            top: 0 !important;
            left: 0 !important;
            width: 100vw !important;
            height: 100vh !important;
            background: #000 !important;
            z-index: 2147483647 !important;
            display: flex !important;
            flex-direction: column !important;
            justify-content: center !important;
            align-items: center !important;
            color: red !important;
            font-family: Arial, sans-serif !important;
            user-select: none !important;
            pointer-events: auto !important;
        `;

        overlay.innerHTML = `
            <div style="text-align: center; padding: 20px;">
                <div style="font-size: 100px; margin-bottom: 20px;">⛔</div>
                <h1 style="color: red; font-size: 36px; margin: 10px 0;">PELANGGARAN TERDETEKSI</h1>
                <h2 style="color: yellow; font-size: 24px; margin: 20px 0;">${reason}</h2>
                <p style="color: white; font-size: 18px; margin: 10px 0;">Anda telah diblokir dari ujian ini.</p>
                <p style="color: white; font-size: 16px; margin: 10px 0;">Pelanggaran Anda telah dicatat oleh sistem.</p>
                <div style="margin-top: 30px;">
                    <div class="spinner-border text-danger" role="status" style="width: 3rem; height: 3rem; border: 4px solid rgba(255,0,0,0.3); border-top-color: red; border-radius: 50%; animation: spin 1s linear infinite;">
                        <span style="position: absolute; width: 1px; height: 1px; padding: 0; margin: -1px; overflow: hidden; clip: rect(0,0,0,0); white-space: nowrap; border: 0;">Loading...</span>
                    </div>
                </div>
                <p style="color: #888; margin-top: 20px; font-size: 14px;">Menghubungi server...</p>
            </div>
        `;

        // Add CSS animation for spinner
        const style = document.createElement('style');
        style.textContent = `
            @keyframes spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
            }
        `;
        overlay.appendChild(style);

        // Prevent ANY interaction on overlay
        overlay.addEventListener('contextmenu', e => { e.preventDefault(); e.stopPropagation(); return false; });
        overlay.addEventListener('keydown', e => { e.preventDefault(); e.stopPropagation(); return false; });
        overlay.addEventListener('keyup', e => { e.preventDefault(); e.stopPropagation(); return false; });
        overlay.addEventListener('click', e => { e.preventDefault(); e.stopPropagation(); return false; });
        overlay.addEventListener('touchstart', e => { e.preventDefault(); e.stopPropagation(); return false; });
        overlay.addEventListener('touchmove', e => { e.preventDefault(); e.stopPropagation(); return false; });
        overlay.addEventListener('touchend', e => { e.preventDefault(); e.stopPropagation(); return false; });

        // Nuclear option: Replace entire document
        document.body.innerHTML = '';
        document.body.appendChild(overlay);

        // Also replace head to remove any scripts that might interfere
        document.title = '⛔ BANNED';

        const payload = JSON.stringify({ session_id: sessionId, reason: reason });

        // Send ban request to server
        fetch(BASE_URL + 'siswa/ban_user', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: payload
        }).then(res => res.json())
            .then(data => {
                console.log('Ban response:', data);
                // Delay for dramatic effect, then force redirect
                setTimeout(() => {
                    window.location.replace(BASE_URL + 'siswa/dashboard'); // Use replace to prevent back button
                }, 3000);
            }).catch(err => {
                console.error('Ban request failed:', err);
                // Still redirect even on error
                setTimeout(() => {
                    window.location.replace(BASE_URL + 'siswa/dashboard');
                }, 3000);
            });
    }

    // ====== RUNTIME DEVTOOLS DETECTION ======

    // Method 1: Window Size Detection (Desktop & Mobile DevTools Panel)
    const sizeThreshold = 160; // Pixel difference threshold

    function detectDevToolsBySize() {
        if (isBanning || devToolsOpen) return;

        const widthDiff = window.outerWidth - window.innerWidth;
        const heightDiff = window.outerHeight - window.innerHeight;

        // Detect if DevTools panel is docked (causes size difference)
        if (widthDiff > sizeThreshold || heightDiff > sizeThreshold) {
            console.warn('DevTools detected via size comparison');
            banUser("Membuka DevTools / Inspect Element (Panel Terdeteksi)");
        }
    }

    // Method 2: Debugger Timing Detection
    function detectDevToolsByDebugger() {
        if (isBanning || devToolsOpen) return;

        const start = performance.now();
        // This will pause execution if DevTools is open
        debugger;
        const end = performance.now();

        // If debugger paused, time difference will be significant
        if (end - start > 100) {
            console.warn('DevTools detected via debugger timing');
            banUser("DevTools terdeteksi (Debugger aktif)");
        }
    }

    // Method 3: Console Detection via toString Override
    const consoleImage = new Image();
    let consoleCheckCount = 0;

    Object.defineProperty(consoleImage, 'id', {
        get: function () {
            consoleCheckCount++;
            // Only trigger after a few checks to avoid false positives during page load
            if (consoleCheckCount > 3 && !isBanning && !devToolsOpen) {
                console.warn('DevTools detected via console inspection');
                banUser("Console Browser terdeteksi terbuka");
            }
            return 'devtools-detector';
        }
    });

    // Method 4: Mobile-specific - Check viewport changes
    function detectMobileDevTools() {
        if (isBanning || devToolsOpen) return;
        if (!('ontouchstart' in window)) return; // Only on mobile

        const vh = window.visualViewport ? window.visualViewport.height : window.innerHeight;
        const wh = window.innerHeight;

        // On mobile, DevTools or keyboard can change viewport height significantly
        if (Math.abs(vh - wh) > 100) {
            // Could be keyboard or DevTools
            console.log('Mobile viewport change detected');
            // Don't ban for keyboard, but log it
            logActivity('Mobile Viewport Change', `VH: ${vh}, WH: ${wh}`);
        }
    }

    // Continuous Detection - Run every second
    setInterval(() => {
        if (!isBanning && !devToolsOpen && !window.isFinishing) {
            detectDevToolsBySize();
            detectMobileDevTools();

            // Console check (less frequent to avoid performance impact)
            if (Math.random() > 0.7) {
                console.log(consoleImage);
                console.clear();
            }

            // Debugger check (least frequent as it's most intrusive)
            if (Math.random() > 0.9) {
                detectDevToolsByDebugger();
            }
        }
    }, 1000);

    // Immediate check on page load
    setTimeout(() => {
        if (window.outerWidth - window.innerWidth > sizeThreshold ||
            window.outerHeight - window.innerHeight > sizeThreshold) {
            alert('⛔ DevTools terdeteksi! Sistem akan memblokir akses.');
            banUser("DevTools sudah terbuka saat memulai ujian");
        }
    }, 500);

    // --- 4. Enhanced Safe Exam Mode Restrictions ---
    document.addEventListener('contextmenu', event => event.preventDefault());

    const protectUi = (e) => {
        e.preventDefault();
        banUser("Mencoba Copy/Paste/Cut");
        return false;
    };
    document.addEventListener('copy', protectUi);
    document.addEventListener('cut', protectUi);
    document.addEventListener('paste', protectUi);

    // Screenshot Detection - Enhanced
    function handleScreenshotAttempt(e, mechanism) {
        e.preventDefault();
        e.stopPropagation();

        // Try to clear clipboard (requires permission, but worth a try)
        if (navigator.clipboard && navigator.clipboard.writeText) {
            navigator.clipboard.writeText('Screenshots are disabled during exam.').catch(err => { });
        }

        banUser("Mencoba Screenshot (" + mechanism + ")");
        return false;
    }

    document.addEventListener('keydown', function (e) {
        // Print Screen key
        if (e.key === 'PrintScreen' || e.keyCode === 44) {
            handleScreenshotAttempt(e, "Print Screen Key");
        }

        // Windows Snipping Tool (Win + Shift + S)
        if (e.key === 'Meta' || e.key === 'OS' || e.keyCode === 91) {
            // Just tracking Meta key press to combine with S might be flaky, 
            // but we can detect the combo if held
        }

        // Win+Shift+S or Cmd+Shift+3/4
        if ((e.metaKey || e.ctrlKey) && e.shiftKey && (e.key === 's' || e.key === 'S')) {
            handleScreenshotAttempt(e, "Snipping Tool Shortcut");
        }

        if (e.metaKey && e.shiftKey && (e.key === '3' || e.key === '4' || e.key === '5')) {
            handleScreenshotAttempt(e, "Mac Screenshot Shortcut");
        }

        // Chrome DevTools Shortcuts
        if (e.key === 'F12') {
            handleScreenshotAttempt(e, "Inspect Element (F12)");
        }
        if (e.ctrlKey && e.shiftKey && (e.key === 'I' || e.key === 'i' || e.key === 'C' || e.key === 'c' || e.key === 'J' || e.key === 'j')) {
            handleScreenshotAttempt(e, "DevTools Shortcut");
        }
        if (e.ctrlKey && (e.key === 'U' || e.key === 'u')) {
            handleScreenshotAttempt(e, "View Source Shortcut");
        }

        // Window Switch (Alt+Tab)
        if (e.altKey && e.key === 'Tab') {
            handleScreenshotAttempt(e, "Alt+Tab Switch");
        }
    });

    // Detect Print Screen via keyup (often more reliable for PrtScn key)
    window.addEventListener('keyup', function (e) {
        if (e.key === 'PrintScreen' || e.keyCode === 44) {
            handleScreenshotAttempt(e, "Print Screen (KeyUp)");
        }
    });

    // Split Screen / Window Resize Detection
    let initialWidth = window.innerWidth;
    let initialHeight = window.innerHeight;
    let resizeTimeout;

    window.addEventListener('resize', function () {
        if (window.isFinishing) return;

        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(function () {
            const widthChange = Math.abs(window.innerWidth - initialWidth);
            const heightChange = Math.abs(window.innerHeight - initialHeight);

            // Significant resize detected (likely split screen or window resize)
            if (widthChange > 100 || heightChange > 100) {
                logActivity('Window Resize Detected', `Width: ${widthChange}px, Height: ${heightChange}px`);
                banUser("Mencoba Split Screen / Resize Window");
            }
        }, 500); // Debounce 500ms
    });

    // Fullscreen Exit Detection
    document.addEventListener('fullscreenchange', function () {
        if (window.isFinishing) return;

        if (!document.fullscreenElement) {
            banUser("Keluar dari mode Fullscreen");
        }
    });

    // New Tab / Window Open Detection
    let windowFocusCount = 0;
    window.addEventListener('load', function () {
        windowFocusCount = 0;
    });

    // Enhanced Blur Detection - ZERO TOLERANCE (untuk telepon masuk, notif, dll)
    let blurCount = 0;
    window.addEventListener("blur", function () {
        if (window.isFinishing) return;
        if (isBanning) return;

        blurCount++;
        const timestamp = new Date().toLocaleTimeString();

        logActivity('Kehilangan Fokus Window', `Blur #${blurCount} at ${timestamp} (Phone call/App switch/Notification)`);

        // IMMEDIATE BAN - No second chances
        console.warn('Window blur detected - immediate ban');
        banUser("Kehilangan Fokus Window (Telepon Masuk/Notifikasi/Aplikasi Lain)");
    });

    // Visibility Change - ZERO TOLERANCE (untuk pindah tab, minimize, lock screen)
    let visibilityChangeCount = 0;
    let lastVisibilityChange = Date.now();

    document.addEventListener("visibilitychange", function () {
        if (window.isFinishing) return;
        if (isBanning) return;

        const now = Date.now();
        const timeSinceLastChange = now - lastVisibilityChange;
        lastVisibilityChange = now;

        if (document.hidden) {
            visibilityChangeCount++;
            const timestamp = new Date().toLocaleTimeString();

            logActivity('Tab/Window Hidden', `Count: ${visibilityChangeCount}, Time: ${timestamp}, Delta: ${timeSinceLastChange}ms`);

            // IMMEDIATE BAN - Zero tolerance
            console.warn('Visibility hidden - immediate ban');
            banUser("Pindah Tab/Window Minimize/Lock Screen/Buka Aplikasi Lain");
        } else {
            // Tab visible again (they came back)
            logActivity('Tab/Window Visible', 'User returned to exam');
            // They're already banned if they left, so this won't execute
        }
    });

    // Page Hide Event - Additional mobile detection
    document.addEventListener("pagehide", function (e) {
        if (window.isFinishing) return;
        if (isBanning) return;

        logActivity('Page Hide Event', 'Mobile app backgrounded or tab closed');
        // This fires when page is about to be unloaded on mobile
        // Can't reliably ban here, but log it
    });

    // Enhanced Focus Detection
    let hasFocus = document.hasFocus();
    setInterval(() => {
        if (window.isFinishing || isBanning) return;

        const currentFocus = document.hasFocus();
        if (hasFocus && !currentFocus) {
            // Lost focus
            hasFocus = false;
            logActivity('Lost Document Focus', 'Focus switched away from exam page');
            // blur event will handle the ban
        } else if (!hasFocus && currentFocus) {
            // Regained focus (but they're probably already banned)
            hasFocus = true;
            logActivity('Regained Document Focus', 'User returned focus');
        }
    }, 500); // Check every 500ms

    // Prevent opening new windows
    window.addEventListener('beforeunload', function (e) {
        if (window.isFinishing) return;

        // This will trigger if user tries to navigate away or close tab
        logActivity('Beforeunload Triggered', 'User attempting to leave page');

        // Note: We can't ban here reliably as the page is unloading
        // But we can log it
        const confirmationMessage = 'Anda sedang mengerjakan ujian! Meninggalkan halaman akan dianggap pelanggaran.';
        e.returnValue = confirmationMessage;
        return confirmationMessage;
    });

    // Detect if window loses focus (user clicked outside)
    let userInteracted = false;
    document.addEventListener('click', function () {
        userInteracted = true;
    });

    // Mobile: Wake Lock & Phone Call Detection - ENHANCED
    if ('wakeLock' in navigator) {
        let wakeLock = null;
        let wakeLockReleaseCount = 0;

        const requestWakeLock = async () => {
            try {
                wakeLock = await navigator.wakeLock.request('screen');
                console.log('✅ Wake Lock acquired - screen will stay on');

                wakeLock.addEventListener('release', () => {
                    wakeLockReleaseCount++;
                    const timestamp = new Date().toLocaleTimeString();

                    console.warn(`Wake Lock released #${wakeLockReleaseCount} at ${timestamp}`);
                    logActivity('Wake Lock Released', `Count: ${wakeLockReleaseCount}, Time: ${timestamp}`);

                    if (!window.isFinishing && !isBanning) {
                        // Wake lock released = screen turned off, phone call, or power button pressed
                        console.error('VIOLATION: Wake Lock released - likely phone call or screen off');
                        banUser("Layar HP Mati/Telepon Masuk/Tombol Power/Screen Lock");
                    }
                });
            } catch (err) {
                console.error('❌ Wake Lock request failed:', err.message);
                logActivity('Wake Lock Failed', err.message);

                // If wake lock not supported or denied, still try to protect
                if (err.name === 'NotAllowedError') {
                    console.warn('Wake Lock not allowed - exam might be vulnerable on this device');
                }
            }
        };

        // Initial request
        requestWakeLock();

        // Re-request when visibility changes (user might have backgrounded app)
        document.addEventListener('visibilitychange', async () => {
            if (document.visibilityState === 'visible' && wakeLock === null) {
                console.log('Visibility changed to visible, re-requesting Wake Lock');
                await requestWakeLock();
            }
        });

        // Monitor wake lock status periodically
        setInterval(() => {
            if (!window.isFinishing && !isBanning) {
                // Check if wake lock is still active
                if (wakeLock === null || wakeLock.released) {
                    logActivity('Wake Lock Status Check', 'Wake Lock is not active');
                    // Don't ban here automatically, let the release event handle it
                }
            }
        }, 5000); // Check every 5 seconds
    } else {
        console.warn('⚠️ Wake Lock API not supported on this device/browser');
        logActivity('Wake Lock Not Supported', 'Device does not support Wake Lock API');
    }

    // Additional Mobile: Battery Status API (for low battery/charging detection)
    if ('getBattery' in navigator) {
        navigator.getBattery().then(battery => {
            console.log(`Battery level: ${battery.level * 100}%`);

            // Monitor for charging events (user might be plugging cable = distraction)
            battery.addEventListener('chargingchange', () => {
                const status = battery.charging ? 'started charging' : 'stopped charging';
                logActivity('Battery Status Change', status);
                // Don't ban for this, just log
            });

            // Monitor for low battery
            battery.addEventListener('levelchange', () => {
                if (battery.level < 0.1) { // Below 10%
                    logActivity('Low Battery Warning', `Battery at ${battery.level * 100}%`);
                    // Don't ban, but user might close exam to charge
                }
            });
        }).catch(err => {
            console.log('Battery API not available:', err.message);
        });
    }

    // Mobile: Screen Orientation Change Detection (rotating phone = distraction)
    if (window.screen && window.screen.orientation) {
        window.screen.orientation.addEventListener('change', (e) => {
            if (window.isFinishing || isBanning) return;

            const orientation = window.screen.orientation.type;
            logActivity('Screen Orientation Changed', `New orientation: ${orientation}`);

            // Don't ban for orientation change, but it's suspicious
            // Could indicate user is trying to take screenshot or share screen
        });
    }

    // Log Start
    logActivity('Mulai Ujian', 'Siswa memulai/melanjutkan ujian');

    console.log('Exam engine initialized successfully');
});
